/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <libpandora/func_scheduler.h>
#include <libpandora/sched_task.h>

time_t func_scheduler::schedule(time_t t, const sched_task_t &st)
{
  if (t <= localTime) {
    //pandora_warning("scheduled in the past!");
    t = localTime + 1;
  }

  m.insert(elt_t(t, st));

#if DEBUG_SCHED
  ++inserted; ++remaining;
#endif
  
  return t;
}

void func_scheduler::remove(time_t t, const sched_task_t &st)
{
  if ( t < localTime ) {
    pandora_error("task should have been removed: " 
		  << t << " < " << localTime);
  }
  
  multi_map_t::iterator ptr;
  multi_map_t::iterator first, last;

  pair<multi_map_t::iterator, multi_map_t::iterator> p
    = m.equal_range(t);

  first = p.first;
  last =  p.second;

  for (ptr = first; ptr != last; ++ptr) {
    if (ptr->second == st) {
      (ptr->second).removed(ptr->first);
      m.erase(ptr);
      break;
    }
  }
#if DEBUG_SCHED
  --remaining; ++removed;
#endif
}

void func_scheduler::expire(time_t t)
{
  multi_map_t::iterator ptr = m.begin();
  while ((ptr != m.end()) && (ptr->first < t)) {
    localTime = ptr->first;
    (ptr->second).expired(localTime);
    m.erase(ptr);
    ptr = m.begin();
#if DEBUG_SCHED
    --remaining; ++expired;
#endif
  }
  localTime = t;
}

void func_scheduler::cleanup(void)
{
  if (m.size() > 0) {
    pandora_warning("tasks have not been collected: " 
		    << m.size() << " remaining");
#if DEBUG_SCHED
    remaining -= m.size(); expired += m.size();
#endif
  }
  m.clear();
}
