/*
 * http_date.c -- Routines for parsing and generating http dates
 * Created: Christopher Blizzard <blizzard@appliedtheory.com>, 16-Aug-1998
 *
 * Copyright (C) 1998 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _XOPEN_SOURCE

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
#include <libpandora/conf/time.h>
#include <ctype.h>
}

#include <libpandora/http_date.h>

#define RFC850_STRFTIME "%A, %d-%b-%y %H:%M:%S GMT"
#define RFC1123_STRFTIME "%a, %d %b %Y %H:%M:%S GMT"
#define ANSIC_STRFTIME "%a %b %d %H:%M:%S %Y"

#include "time_days.h"

inline static time_t
__mktime(struct tm *l_tm)
{
  return time_days[l_tm->tm_year - 70][l_tm->tm_mon] 
    + (l_tm->tm_mday-1) * 86400 + l_tm->tm_hour * 3600 
    + l_tm->tm_min * 60 + l_tm->tm_sec;
}

static int month_hash[32] = 
  {  0,  0,  7,  0,  0,  0,  0,  0,
     0,  0,  0,  3,  0,  0,  0,  2,
     6,  0,  5,  0,  9,  8,  4,  0,
     0, 11,  1, 10,  0,  0,  0,  0 };

inline static int
month_from_string_short(const char *s)
{
  int hash = (unsigned char) s[0] + (unsigned char) s[2];
  hash = (hash << 1) + (unsigned char) s[1];
  hash = (hash & 63) >> 1;
  return month_hash[hash];
}

/* 
 *  date formats come in one of the following three flavors according to
 *  rfc 2068 and later drafts:
 *  
 *  Sun, 06 Nov 1994 08:49:37 GMT     ; RFC 822, updated by RFC 1123
 *  Sunday, 06-Nov-1994, 08:49:37 GMT ; RFC 850, updated by RFC 1036
 *  Sun Nov  6 08:49:37 1994          ; ANSI C's asctime() format
 *
 *  the first format is preferred however all must be supported.
*/

time_t
http_date_to_time(const char *a_date)
{
  struct tm       l_tm_time;
  time_t          l_return = 0;
  char            l_buf[12];
  const char     *l_start_date = NULL;
  int             i = 0;

  /* make sure we can use it */
  if (!a_date)
    return -1;
  memset(&l_tm_time, 0, sizeof(struct tm));
  memset(l_buf, 0, 12);
  /* try to figure out which format it's in */
  /* rfc 1123 */
  if (a_date[3] == ',')
    {
      if (strlen(a_date) < 29)
	goto error;
      /* make sure that everything is legal */
      if (a_date[4] != ' ')
	goto error;
      /* 06 */
      if ((isdigit(a_date[5]) == 0) ||
	  (isdigit(a_date[6]) == 0))
	goto error;
      /* Nov */
      if ((l_tm_time.tm_mon = month_from_string_short(&a_date[8])) < 0)
	goto error;
      /* 1994 */
      if ((isdigit(a_date[12]) == 0) ||
	  (isdigit(a_date[13]) == 0) ||
	  (isdigit(a_date[14]) == 0) ||
	  (isdigit(a_date[15]) == 0))
	goto error;
      if (a_date[16] != ' ')
	goto error;
      /* 08:49:37 */
      if ((isdigit(a_date[17]) == 0) ||
	  (isdigit(a_date[18]) == 0) ||
	  (a_date[19] != ':') ||
	  (isdigit(a_date[20]) == 0) ||
	  (isdigit(a_date[21]) == 0) ||
	  (a_date[22] != ':') ||
	  (isdigit(a_date[23]) == 0) ||
	  (isdigit(a_date[24]) == 0))
	goto error;
      if (a_date[25] != ' ')
	goto error;
      /* GMT */
      if (strncmp(&a_date[26], "GMT", 3) != 0)
	goto error;
      /* ok, it's valid.  Do it */
      /* parse out the day of the month */
      l_tm_time.tm_mday += (a_date[5] - 0x30) * 10;
      l_tm_time.tm_mday += (a_date[6] - 0x30);
      /* already got the month from above */
      /* parse out the year */
      l_tm_time.tm_year += (a_date[12] - 0x30) * 1000;
      l_tm_time.tm_year += (a_date[13] - 0x30) * 100;
      l_tm_time.tm_year += (a_date[14] - 0x30) * 10;
      l_tm_time.tm_year += (a_date[15] - 0x30);
      l_tm_time.tm_year -= 1900;
      /* parse out the time */
      l_tm_time.tm_hour += (a_date[17] - 0x30) * 10;
      l_tm_time.tm_hour += (a_date[18] - 0x30);
      l_tm_time.tm_min  += (a_date[20] - 0x30) * 10;
      l_tm_time.tm_min  += (a_date[21] - 0x30);
      l_tm_time.tm_sec  += (a_date[23] - 0x30) * 10;
      l_tm_time.tm_sec  += (a_date[24] - 0x30);
      /* ok, generate the result */
    }
  /* ansi C */
  else if (a_date[3] == ' ')
    {
      if (strlen(a_date) < 24)
	goto error;
      /* Nov */
      if ((l_tm_time.tm_mon =
	   month_from_string_short(&a_date[4])) < 0)
	goto error;
      if (a_date[7] != ' ')
	goto error;
      /* "10" or " 6" */
      if (((a_date[8] != ' ') && (isdigit(a_date[8]) == 0)) ||
	  (isdigit(a_date[9]) == 0))
	goto error;
      if (a_date[10] != ' ')
	goto error;
      /* 08:49:37 */
      if ((isdigit(a_date[11]) == 0) ||
	  (isdigit(a_date[12]) == 0) ||
	  (a_date[13] != ':') ||
	  (isdigit(a_date[14]) == 0) ||
	  (isdigit(a_date[15]) == 0) ||
	  (a_date[16] != ':') ||
	  (isdigit(a_date[17]) == 0) ||
	  (isdigit(a_date[18]) == 0))
	goto error;
      if (a_date[19] != ' ')
	goto error;
      /* 1994 */
      if ((isdigit(a_date[20]) == 0) ||
	  (isdigit(a_date[21]) == 0) ||
	  (isdigit(a_date[22]) == 0) ||
	  (isdigit(a_date[23]) == 0))
	goto error;
      /* looks good */
      /* got the month from above */
      /* parse out the day of the month */
      if (a_date[8] != ' ')
	l_tm_time.tm_mday += (a_date[8] - 0x30) * 10;
      l_tm_time.tm_mday += (a_date[9] - 0x30);
      /* parse out the time */
      l_tm_time.tm_hour += (a_date[11] - 0x30) * 10;
      l_tm_time.tm_hour += (a_date[12] - 0x30);
      l_tm_time.tm_min  += (a_date[14] - 0x30) * 10;
      l_tm_time.tm_min  += (a_date[15] - 0x30);
      l_tm_time.tm_sec  += (a_date[17] - 0x30) * 10;
      l_tm_time.tm_sec  += (a_date[18] - 0x30);
      /* parse out the year */
      l_tm_time.tm_year += (a_date[20] - 0x30) * 1000;
      l_tm_time.tm_year += (a_date[21] - 0x30) * 100;
      l_tm_time.tm_year += (a_date[22] - 0x30) * 10;
      l_tm_time.tm_year += (a_date[23] - 0x30);
      l_tm_time.tm_year -= 1900;
      /* generate the result */
    }
  /* must be the 1036... */
  else
    {
      /* check to make sure we won't rn out of any bounds */
      if (strlen(a_date) < 11)
	goto error;
      while (a_date[i])
	{
	  if (a_date[i] == ' ')
	    {
	      l_start_date = &a_date[i+1];
	      break;
	    }
	  i++;
	}
      /* check to make sure there was a space found */
      if (l_start_date == NULL)
	goto error;
      /* check to make sure that we don't overrun anything */
      if (strlen(l_start_date) < 22)
	goto error;
      /* make sure that the rest of the date was valid */
      /* 06- */
      if ((isdigit(l_start_date[0]) == 0) ||
	  (isdigit(l_start_date[1]) == 0) ||
	  (l_start_date[2] != '-'))
	goto error;
      /* Nov */
      if ((l_tm_time.tm_mon =
	   month_from_string_short(&l_start_date[3])) < 0)
	goto error;
      /* -94 */
      if ((l_start_date[6] != '-') ||
	  (isdigit(l_start_date[7]) == 0) ||
	  (isdigit(l_start_date[8]) == 0))
	goto error;
      if (l_start_date[9] != ' ')
	goto error;
      /* 08:49:37 */
      if ((isdigit(l_start_date[10]) == 0) ||
	  (isdigit(l_start_date[11]) == 0) ||
	  (l_start_date[12] != ':') ||
	  (isdigit(l_start_date[13]) == 0) ||
	  (isdigit(l_start_date[14]) == 0) ||
	  (l_start_date[15] != ':') ||
	  (isdigit(l_start_date[16]) == 0) ||
	  (isdigit(l_start_date[17]) == 0))
	goto error;
      if (l_start_date[18] != ' ')
	goto error;
      if (strncmp(&l_start_date[19], "GMT", 3) != 0)
	goto error;
      /* looks ok to parse */
      /* parse out the day of the month */
      l_tm_time.tm_mday += (l_start_date[0] - 0x30) * 10;
      l_tm_time.tm_mday += (l_start_date[1] - 0x30);
      /* have the month from above */
      /* parse out the year */
      l_tm_time.tm_year += (l_start_date[7] - 0x30) * 10;
      l_tm_time.tm_year += (l_start_date[8] - 0x30);
      /* check for y2k */
      if (l_tm_time.tm_year < 70)
	l_tm_time.tm_year += 100;
      /* parse out the time */
      l_tm_time.tm_hour += (l_start_date[10] - 0x30) * 10;
      l_tm_time.tm_hour += (l_start_date[11] - 0x30);
      l_tm_time.tm_min  += (l_start_date[13] - 0x30) * 10;
      l_tm_time.tm_min  += (l_start_date[14] - 0x30);
      l_tm_time.tm_sec  += (l_start_date[16] - 0x30) * 10;
      l_tm_time.tm_sec  += (l_start_date[17] - 0x30);
      /* generate the result */
    }
  return __mktime(&l_tm_time);

 error:
  //cerr << "invalid date: " << a_date << "\n";
  return -2;
}

int
http_time_to_date(char *buf, size_t len, time_t t)
{
  struct tm *gmt = gmtime(&t);

  buf[0] = '\0';
  return (strftime(buf, len-1, RFC1123_STRFTIME, gmt) > 0);
}
