/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef LCS_H
#define LCS_H

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
#include <libpandora/conf/string.h>
	   }

template <class T>
struct lcs_t {
  T *elts;
  size_t nelt;
};

struct lcs_op_t {
  enum operation_t {lcs_undef = -1, lcs_match, lcs_ins, lcs_del};
  operation_t 	type;
  int 		src, dst;

  lcs_op_t(void) : type(lcs_undef), src(-1), dst(-1) {}
  lcs_op_t(const lcs_op_t &x) : type(x.type), src(x.src), dst(x.dst) {}
  lcs_op_t &operator=(const lcs_op_t &x) {
    type = (x.type); src = x.src; dst = x.dst;
    return *this;
  }
  ~lcs_op_t(void) {}
  void print(void) {
    switch(type) {
    case lcs_match: cerr << "match:\t\t"; break;
    case lcs_ins: cerr << "inserted:\t"; break;
    case lcs_del: cerr << "deleted:\t"; break;
    default: break;
    }
    cerr << src << " -> " << dst << endl;
  }
};


struct lcs_script_t {
  int nops, size;
  lcs_op_t *ops;

  lcs_script_t(void) : nops(0), size(0), ops(NULL) { }
  ~lcs_script_t(void) { __DELETE_ARRAY(ops); }
  void init(int s) {
    size = s;
    __DELETE_ARRAY(ops);
    ops = new lcs_op_t[size];
  }
  void add(lcs_op_t &op) {
    if (nops < size) {
      ops[nops] = op;
      ++nops;
    } else {
      pandora_warning("too many ops recorded");
    }
  }
  void print(void) {
    for (int i = 0; i < nops; ++i) ops[i].print();
  }
};


template <class T>
void lcs_print(const lcs_t<T> *a, const lcs_t<T> *b, 
	       bool (*equal)(const T &, const T &),
	       lcs_script_t *script)
{
  if (a == NULL || b == NULL) return;

  register int m = a->nelt, n = b->nelt;
  T *A = a->elts;
  T *B = b->elts;

  int **L = new int* [m+1];
  for (int k = 0; k <= m; ++k) {
    L[k] = new int[n+1];
    memset((char *)L[k], 0, (n+1)*sizeof(int));
  }

  for (int i = m-1; i >= 0; i--)
    for (int j = n-1; j >= 0; j--)
      L[i][j] = (equal(A[i], B[j]))  
	? 1 + L[i+1][j+1] 
	: pandora_max(L[i+1][j], L[i][j+1]);

  script->init(m + n - L[m][n]);
  do {
    int i = 0, j = 0;
    while ((i < m) && (j < n)) {
      lcs_op_t op;
      if (equal(A[i], B[j])) {
	op.type = lcs_op_t::lcs_match; op.src = i; op.dst = j;
	i++; j++;
      } else if (L[i+1][j] > L[i][j+1]) {
	op.type = lcs_op_t::lcs_del; op.src = i;
	i++;
      } else {
	op.type = lcs_op_t::lcs_ins; op.dst = j;
	j++;
      }
      script->add(op);
    }
    
    if (i == m) {
      for (/* empty */; j < n; ++j) {
	lcs_op_t op;
	op.type = lcs_op_t::lcs_ins; op.src = j;
	script->add(op);
      }
    } else if (j == n) {
      for (/* empty */; i < m; ++i) {
	lcs_op_t op;
	op.type = lcs_op_t::lcs_del; op.src = i;
	script->add(op);
      }
    } 
  } while(0);

  for (int k = 0; k <= m; ++k) __DELETE_ARRAY(L[k]);
  __DELETE_ARRAY(L);
}

#endif /* LCS_H */
