/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef MAP_TREE_H
#define MAP_TREE_H

#include <libpandora/global.h>

#include <libpandora/map.h>
#include <libpandora/association.h>

#define GENERIC	template<class Key, class Val> inline
#define MAPTREE	MapTree<Key, Val>

template<class Key, class Val>
class MapTree
{
  static Val _undef;
  MAPTREE *up;
  Map<Key,MAPTREE *> next;
  Val value;

public:
  MapTree(void) : up(NULL) { }
 virtual ~MapTree(void);

private:
  MapTree(MAPTREE *u) : up(u) { }

public:
  Val	      &atOrNil(const Key *key, int keysize) const;
  Val	      &at(const Key *key, int keysize) const;
  void	       atPut(const Key *key, int keysize, const Val &anObject);
  bool	       includesKey(const Key *key, int keysize) const;
  void	       removeKey(const Key *key, int keysize);
};


//
// INLINE IMPLEMENTATION
//

GENERIC MAPTREE::~MapTree(void) 
{
  while(next.size() > 0) { 
    Key key;
    MAPTREE *node; 
    keysValuesDo(next, key, node) {
      next.removeKey(key);
      __DELETE(node);
    }
  }
}

// Answer the association with the given key.

GENERIC Val &MAPTREE::atOrNil(const Key *key, int keysize) const
{
  pandora_assert(key != NULL);
  if (keysize == 0) return (Val &)value;
  MAPTREE *node = next.atOrNil(*key);
  if (node == NULL) return (Val &)_undef;
  return node->atOrNil(key+1, keysize-1);
}

GENERIC Val &MAPTREE::at(const Key *key, int keysize) const
{
  pandora_assert(key != NULL);
  if (keysize == 0) return (Val &)value;
  MAPTREE *node = next.atOrNil(*key);
  pandora_assert (node != NULL);
  return node->at(key+1, keysize-1);
}


// Set the value at key to be anObject.  If key is not found, create a new
// entry for key and set is value to anObject.

GENERIC void MAPTREE::atPut(const Key *key, int keysize, const Val &anObject)
{
  if (keysize == 0) {
    value = anObject;
    return;
  }

  MAPTREE *node = next.atOrNil(*key);
  if (node == NULL) {
    node = new MAPTREE(this);
    next.atPut(*key, node);
  }
  node->atPut(key+1, keysize-1, anObject);
}


// Answer whether the receiver has a key equal to the argument, key.

GENERIC bool MAPTREE::includesKey(const Key *key, int keysize) const
{
  if (keysize == 0) return true;
  MAPTREE *node = next.atOrNil(*key);
  if (node == NULL) return false;
  return node->includesKey(key+1, keysize-1);
}


// Remove key (and its associated value) from the map.

GENERIC void MAPTREE::removeKey(const Key *key, int keysize)
{
  if (keysize == 0) return;
  MAPTREE *node = next.atOrNil(*key);
  pandora_assert(node != NULL);
  node->removeKey(key+1, keysize-1);
  if ((node->next).size() == 0) __DELETE(node);
}
template<class Key, class Val> Val MAPTREE::_undef;
#undef GENERIC
#undef MAPTREE


#endif /* MAP_TREE_H */
