/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef MULTIMAP_H
#define MULTIMAP_H

#include <libpandora/global.h>

#include <libpandora/map.h>

#define GENERIC		template<class Key, class Val> inline
#define MULTIMAP	MultiMap<Key, Val>
#define SET		Set<Val>

template<class Key, class Val>
class MultiMap
{
public:
  Map<Key, Set<Val> *> _data;


public:
  MultiMap(void) { }
  virtual ~MultiMap(void);

public:
  int   at(const Key &key, Val *vals, int maxsize);
  int   atOrNil(const Key &key, Val *vals, int maxsize);
  void	atPut(const Key &key, const Val &val);
  bool  includes(const Key &key, const Val &val) const;
  bool	includesKey(const Key &key) const;
  void	removeKey(const Key &key);
  long  size(void) const			{ return _data.size(); }
};


//
// INLINE IMPLEMENTATION
//

GENERIC MULTIMAP::~MultiMap(void)
{
  Key key;

  while(_data.size() > 0) {
    keysDo(_data, key) removeKey(key);
  }
}

GENERIC void MULTIMAP::atPut(const Key &key, const Val &val)
{
  SET *s = _data.atOrNil(key);
  if (s == NULL) {
    s = new SET;
    _data.atPut(key, s);
  }
  s->add(val);
}

GENERIC  int MULTIMAP::at(const Key &key, Val *vals, int maxsize)
{
  if (vals == NULL) return -1;
  if (maxsize <= 0) return maxsize;
  SET *s = _data.atOrNil(key);
  pandora_assert (s != NULL);

  int n = 0;
  Val val;
  elementsDo(*s, val) {
    vals[n++] = val;
    if (n == maxsize) break;
  }
  return n;
}

GENERIC  int MULTIMAP::atOrNil(const Key &key, Val *vals, int maxsize)
{
  if (vals == NULL) return -1;
  if (maxsize <= 0) return maxsize;
  SET *s = _data.atOrNil(key);
  if (s == NULL) return 0;

  int n = 0;
  Val val;
  elementsDo(*s, val) {
    vals[n++] = val;
    if (n == maxsize) break;
  }
  return n;
}

GENERIC bool MULTIMAP::includes(const Key &key, const Val &val) const
{
  const SET *s = _data.atOrNil(key);
  if (s == NULL) return false;
  return s->includes(val);
}

GENERIC bool MULTIMAP::includesKey(const Key &key) const
{
  return _data.includesKey(key);
}

GENERIC void  MULTIMAP::removeKey(const Key &key)
{
  SET *s = _data.removeKey(key);
  __DELETE(s);
}

#undef GENERIC
#undef MULTIMAP
#undef SET

#if 0
#define multiKeysDo(MULTIMAP, KEY) keysDo(_data, KEY)
#define multiValuesDo(MULTIMAP, VAL) 
#define multiKeysValuesDo(MULTIMAP, KEY, VAL)
#endif

#endif /* MULTIMAP_H */
