/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <netdb.h>
#include <errno.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/fcntl.h>
#include <sys/select.h>
#include <stdio.h>
#include <libpandora/conf/string.h>
#include <libpandora/conf/poll.h>
#include <libpandora/conf/time.h>

#ifdef sun
int gethostname(char *name, int namelen);
#endif
#ifndef HAVE_HSTRERROR	
extern const char *hstrerror (int herr);
#endif

   }

#include <libpandora/thread.h>
#include <libpandora/netutil.h>
#include <libpandora/error.h>

extern ostream &operator<< (ostream &, const struct in_addr & );

static Mutex gethost_mutex;

int openserver(int server_port, bool inet, bool stream)
{
  int fd = socket((inet ? AF_INET : AF_LOCAL), 
		  (stream ? SOCK_STREAM : SOCK_DGRAM), 
		  0);
  if (fd < 0) { 
    pandora_pwarning("socket");
    return -1;
  }


  char unix_path[64];

  if (inet) {
    int one = 1;
    if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, 
		   (char *)&one, sizeof(int)) < 0) {
      pandora_pwarning("setsockopt");
      close(fd);
      return -1;
    }
  } else { 
    if (snprintf(unix_path, sizeof(unix_path),
		 "/tmp/.pandora/%d", server_port) < 0) {
      pandora_pwarning("snprintf");
      close(fd);
      return -1;
    }
    mode_t oldm = umask(0);
    (mkdir("/tmp/.pandora", 01777) < 0) 
      && unlink(unix_path);
    umask(oldm);
  }

  struct sockaddr_in inet_addr;
  struct sockaddr_un unix_addr;
  struct sockaddr *server_addr;
  int length;

  if (inet) {
    length = sizeof(struct sockaddr_in);
    server_addr = (sockaddr *)&inet_addr;
    memset(&inet_addr, 0, length);
    inet_addr.sin_family = AF_INET;
    inet_addr.sin_port = htons(server_port);
    inet_addr.sin_addr.s_addr = INADDR_ANY;
  } else {
    server_addr = (sockaddr *)&unix_addr;
    memset(&unix_addr, 0, sizeof(sockaddr_un));
    unix_addr.sun_family = AF_LOCAL;
    strncpy(unix_addr.sun_path, unix_path, sizeof(unix_addr.sun_path));
    length = SUN_LEN(&unix_addr);
  }

  if (bind(fd, server_addr, length) < 0) {
    pandora_pwarning("bind");
    close(fd);
    return -1;
  }

#ifndef SO_MAXCONN
#define SO_MAXCONN	128
#endif  

  if (stream) {
    if (listen(fd, SO_MAXCONN) < 0) {
      pandora_pwarning("listen");
      close(fd);
      return -1;
    }
  }

  return fd;
}

int openclient(int server_port, in_addr_t ipnum, 
	       bool inet, bool stream, 
	       int ct, in_addr_t ipbind)
{
  int fd, length;
  int err = 0;
  socklen_t len = sizeof(int);
  char unix_path[64];

  struct sockaddr_in inet_addr;
  struct sockaddr_un unix_addr;
  struct sockaddr *server_addr;
  
  if ((fd = socket((inet ? AF_INET : AF_LOCAL),
		   (stream ? SOCK_STREAM : SOCK_DGRAM), 0)) < 0) {
    pandora_pwarning("socket");
    return -1;
  }

  if (inet) {
    if (ipbind != 0) {
      memset(&inet_addr, 0, sizeof(struct sockaddr_in));
      inet_addr.sin_family = AF_INET;
      inet_addr.sin_port = 0;
      inet_addr.sin_addr.s_addr = ipbind;
      
      if (bind(fd, (sockaddr *)&inet_addr, sizeof(sockaddr_in)) < 0) {
	pandora_pwarning("bind");
      }
    }
  } else {
    if (snprintf(unix_path, sizeof(unix_path),
		 "/tmp/.pandora/%d", server_port) < 0) {
      pandora_pwarning("snprintf");
      close(fd);
      return -1;
    }
   }

  int flags = fcntl(fd, F_GETFL);
  if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) < 0) {
    pandora_pwarning("fcntl");
    goto failed;
  }
  
  if (inet) {    
    length = sizeof(sockaddr_in);
    server_addr = (sockaddr *)&inet_addr;
    memset(&inet_addr, 0, length);
    inet_addr.sin_family = AF_INET;
    inet_addr.sin_port = htons(server_port);
    inet_addr.sin_addr.s_addr = ipnum;
  } else {
    server_addr = (sockaddr *)&unix_addr;
    memset(&unix_addr, 0, sizeof(sockaddr_un));
    unix_addr.sun_family = AF_LOCAL;
    strncpy(unix_addr.sun_path, unix_path, sizeof(unix_addr.sun_path));
    length = SUN_LEN(&unix_addr);
  }
  
  if ((connect(fd, server_addr, length) < 0)
      && (errno != EINPROGRESS)) {
    err = errno;
    if (errno != 0) goto failed;
  }

  do {
    struct pollfd ufd;
    memset((char *)&ufd, 0, sizeof(ufd));
    ufd.fd = fd;
    ufd.events = POLLOUT;
    int timeout = (ct > 0) ? 1000*ct : -1;

    switch(poll(&ufd, 1, timeout)) {
    case 0: 	err = ETIMEDOUT; goto failed;
    case 1: 	break;
    default:  	pandora_pwarning("poll"); goto failed;
    }
  } while (0);

  fcntl(fd, F_SETFL, flags);

  if (getsockopt(fd, SOL_SOCKET, SO_ERROR, (char *)&err, &len) < 0) {
    pandora_pwarning("getsockopt");
    goto failed;
  }

  if (err != 0) goto failed;
  
  return fd;
  
 failed:
  close(fd);
  if (inet) {
    struct in_addr a;
    a.s_addr = ipnum;
    pandora_pwarning("failed connect to " << a << ":" << server_port);
  } else {
    pandora_pwarning("failed connect to unix:" << server_port);
  }
  return -1; 
}

int net_writeblock(int fd, const char *buffer, int nbytes)
{
  int n;
  int offset;
  
  for (offset = 0; offset < nbytes; offset += n) {
    if ((n = send(fd, buffer + offset, nbytes - offset, 0)) < 0) {
      if (errno == EINTR) {
	n = 0;
      } else {
	pandora_pwarning("net_writeblock");
	return -1;
      }
    } else if (n==0) break;
  }

  return offset;
}

int net_readblock(int fd, char *buffer, int nbytes)
{
  register int n;
  register int offset;
  
  for (offset = 0; offset < nbytes; offset += n) {
    if ((n = recv(fd, buffer + offset, nbytes - offset, MSG_NOSIGNAL)) < 0) {
      if (errno == EINTR) {
	n = 0;
      } else {
	pandora_pwarning("net_readblock");
	return -1;
      }
    } else if (n==0) break;
  }
  
  return offset;
}

in_addr_t get_addr(const char *server_node)
{
  struct hostent *node_ptr;
  
  if (server_node == NULL) return 0;

#if 1
  int l = strlen(server_node);
  char last = server_node[l-1];
  if ((last >= '0') & (last <= '9')) {
    struct in_addr addr;
    if (inet_aton(server_node, &addr) > 0) 
      return addr.s_addr;
  }
#endif

  // gethostbyname is NOT reentrant...
  gethost_mutex.lock();
  node_ptr = gethostbyname(server_node);
  gethost_mutex.unlock();
  
  if (node_ptr == NULL) {
    pandora_warning("open_client: " << server_node << ": dns lookup failure");
    return 0;
  }
  
  return *((in_addr_t *) node_ptr->h_addr);  
}

int timeout_read(int fd, int timeout)
{
  if (timeout < 0) return 1;

  fd_set tset;  
  timeval tv;
  tv.tv_sec = timeout;
  tv.tv_usec = 0;
  
  FD_ZERO(&tset);
  FD_SET(fd, &tset);

  int n = select(fd+1, &tset, NULL, NULL, &tv);

  switch (n) {
  case 1:  							break;
  case 0:  pandora_warning("timeout_read: timeout expired ");  	break;
  default: pandora_pwarning("select"); 				break;
  }

  return n;
}

void fdopen_socket(int fd, FILE **in, FILE **out)
{
  if ((in == NULL) | (out == NULL) | (fd < 0)) return;

  *in = *out = NULL;

  if ((*in = fdopen(fd, "r")) == NULL) {
    close(fd);
    pandora_pwarning("fdopen");
    return;
  }

  if ((*out = fdopen(fd, "w")) == NULL) {
    close(fd);
    fclose(*in);
    *in = NULL;
    pandora_pwarning("fdopen");
    return;
  }
  
  setvbuf(*in, NULL, _IOFBF, BUFSIZ);
  setvbuf(*out, NULL, _IONBF, 0);
}

void fdopen_socket(int fd, FILE **in)
{
  if ((in == NULL) | (fd < 0)) return;

  *in = NULL;

  if ((*in = fdopen(fd, "r")) == NULL) {
    close(fd);
    pandora_pwarning("fdopen");
    return;
  }

  setvbuf(*in, NULL, _IOFBF, BUFSIZ);
}
