/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef PACKET_DATA_H
#define PACKET_DATA_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
}

#include <libpandora/util.h>
#include <libpandora/error.h>
#include <libpandora/serialize.h>

class packetData {
private:
  bool   priv;
  char   *data;
  size_t length;
  size_t capacity;
  size_t read_offset;

public:
  inline packetData(void);
  inline packetData(int);
  inline packetData(const char *d, const size_t l);
  inline packetData(const packetData &x);
  inline packetData &operator=(const packetData &x);
  inline ~packetData(void);

  inline void set(const char *d, const size_t l);
  inline void setEmpty(const size_t c);
  inline void setNoCopy(packetData &x);
  inline void setNoCopy(char *d, const size_t l);
  inline void doCopy(void);

  inline void newbuf(int size);
  inline void clear(void);
  inline void cleanup(void);

  inline void grow(void);
  inline void grow(size_t c);
  inline void growUp(size_t c);
  inline void shrink(size_t s);

  inline void write(const char *d, const size_t l);
  inline void atWrite(const char *d, const size_t l, const size_t pos);

  inline void move(const int s);
  inline void go(const size_t s) ;
  inline void read(char *d, const size_t l);
  inline void atRead(char *d, const size_t l, const size_t pos) const;
  inline char &operator[](const int i);
  inline u_short getShort(void);
  inline u_int32_t getLong(void);

  inline bool isEmpty(void) const;
  inline char *getData(void);
  inline size_t getLength(void) const;
  inline size_t getCapacity(void) const;
  inline size_t getSpace(void) const;

  template <class T> void writeObject(const T &arg);
  template <class T> void readObject(T &arg);

  friend void serialize(char *, size_t &, const size_t , const packetData *);
  friend void unserialize(const char *, size_t &, packetData *);
};


inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const packetData *var)
{
  int len = var->length - var->read_offset;
  serialize(str, count, maxlen, &len);
  if (len > 0) {
    pandora_assert(maxlen > (count + len));
    memcpy(str + count, (char *)var->data + var->read_offset, len);
    count += len;
  }
}

inline void unserialize(const char *str, size_t &count, packetData *var)
{
  int len;
  unserialize(str, count, &len);
  if (len > 0) {
    var->set((const char *)str + count, len);
    count += len;
  }
}

packetData::packetData(void) 
  : priv(false), data(NULL), length(0), capacity(0), read_offset(0)
{
}

packetData::packetData(int c) 
  : priv(true), length(0), capacity(c), read_offset(0)
{
  data = (char *)xmalloc((c+2)*sizeof(char));
  data[0] = data[1] = '\0';
}

packetData::packetData(const char *d, const size_t l) 
  : priv(true), length(l), capacity(l), read_offset(0)
{
  data = (char *)xmalloc((l+2)*sizeof(char));
  memcpy(data, d, l * sizeof(char));
  data[l] = data[l+1] = '\0';
}

packetData::packetData(const packetData &x) 
  : priv(true), length(x.length), capacity(x.capacity), 
    read_offset(x.read_offset)
{
  if (x.data == NULL) {
    data = NULL;
  } else {
    data = (char *)xmalloc((length+2)*sizeof(char));
    memcpy(data, (const char *) x.data, length * sizeof(char));
    data[length] = data[length+1] = '\0';
  }
}

packetData &packetData::operator=(const packetData &x) 
{
  length = x.length; capacity = x.capacity;
  read_offset = x.read_offset;
  if ((data != NULL) & priv) xfree(data);
  if (x.data == NULL) {
    data = NULL;
  } else {
    data = (char *)xmalloc((length+2)*sizeof(char));
    memcpy(data, (const char *) x.data, length * sizeof(char));
    data[length] = data[length+1] = '\0';
  }
  priv = true;
  return *this;
}

packetData::~packetData(void) 
{
#if 0
  if (!priv) { 
    pandora_assert(length > 0); 
    cout << length << "\n"; 
  }
#endif
  cleanup();
}

void packetData::set(const char *d, const size_t l) 
{
  newbuf(l);
  memcpy(data, d, l * sizeof(char));
  data[l] = data[l+1] = '\0';
  length = l; read_offset = 0;
}

void packetData::setEmpty(const size_t c) 
{
  newbuf(c);
  data[0] = data[1] = '\0';
  length = read_offset = 0;
}

void packetData::setNoCopy(packetData &x) 
{
  cleanup();
  priv = x.priv; data = x.data; length = x.length; capacity = x.capacity;
  read_offset = x.read_offset;
  x.priv = false;
}

void packetData::setNoCopy(char *str, const size_t l) 
{
  cleanup();
  data = str;
  length = capacity = l;
  priv = false;
}

void packetData::doCopy(void) 
{
  if (priv) return;
  const char *d = data;
  data = (char *)xmalloc((length+2)*sizeof(char));
  memcpy(data, d, length * sizeof(char));
  data[length] = data[length+1] = '\0';
  capacity = length;
  priv = true;
}

void packetData::newbuf(int size) 
{
  if (size < (int)capacity) return;
  data = (priv 
	  ? (char *) xrealloc(data, size + 2)
	  : (char *) xmalloc(size+2));
  capacity = size;
  priv = true;
}

void packetData::clear(void) 
{
  length = read_offset = 0;
  data[0] = data[1] = '\0';
}

void packetData::cleanup(void) 
{
  if (priv) __FREE(data); 
  else      data = NULL;
  length = capacity = read_offset = 0;
}

void packetData::grow(void) 
{
  newbuf(capacity * 2);
}

void packetData::grow(size_t c) 
{
  newbuf(capacity+c);
}

void packetData::growUp(size_t c) 
{
  newbuf(c);
}

void packetData::shrink(size_t s) 
{
  if ((s + read_offset) > length) return;
  length = s + read_offset;
  data[length] = data[length+1] = '\0';
}

void packetData::write(const char *d, const size_t l) 
{
  if (d == NULL) return;
  pandora_assert((length + l) <= capacity);
  memcpy((char *)(data + length), (char *)d, l * sizeof(char));
  length += l;
  data[length] = data[length+1] = '\0';    
}

void packetData::atWrite(const char *d, const size_t l, const size_t pos) 
{
  if (d == NULL) return;
  pandora_assert((pos + l) <= capacity);
  memcpy((char *)(data +  pos), (char *)d, l * sizeof(char));
  length = pandora_max(length, pos + l);
  data[length] = data[length+1] = '\0';    
}

void packetData::move(const int s) 
{
  pandora_assert(((int)(read_offset+s) >= 0) && ((read_offset+s) <= length));
  read_offset += s;
}

void packetData::go(const size_t s) 
{
  pandora_assert(s < length);
  read_offset = s;
}

void packetData::read(char *d, const size_t l)
{
  pandora_assert((l + read_offset) < length);
  memcpy((char *)d, (char *)(data+read_offset), l * sizeof(char));
  read_offset += l;
}

void packetData::atRead(char *d, const size_t l, const size_t pos) const
{
  pandora_assert((l + pos) < length);
  memcpy((char *)d, (char *)(data + pos), l * sizeof(char));
}

char &packetData::operator[](const int i)
{
  pandora_assert( data != NULL );
  pandora_assert((read_offset +i) <= length);
  return *(data+read_offset+i);
}

u_short packetData::getShort(void)
{
  pandora_assert((length - read_offset) >= 2);
  register char *t_cp = (char *)(data+read_offset);
  read_offset += 2;
  return ((u_short)t_cp[0] << 8) | ((u_short)t_cp[1]); 
}

u_int32_t packetData::getLong(void)
{
  pandora_assert((length - read_offset) >= 4);
  register char *t_cp = (char *)(data+read_offset);
  read_offset += 4;
  return ((u_int32_t)t_cp[0] << 24) | ((u_int32_t)t_cp[1] << 16)
    | ((u_int32_t)t_cp[2] << 8) | ((u_int32_t)t_cp[3]);
}

bool packetData::isEmpty(void) const 
{ 
  return (data == NULL); 
}

char *packetData::getData(void) 
{ 
  return (data + read_offset); 
}

size_t packetData::getLength(void) const
{
  return (length - read_offset); 
}

size_t packetData::getCapacity(void) const
{
  return capacity; 
}

size_t packetData::getSpace(void) const
{
  return (capacity - length); 
}

template <class T> 
void packetData::writeObject(const T &arg)
{
  serialize(data, length, capacity, &arg);
  
}

template <class T> 
void packetData::readObject(T &arg)
{
  unserialize(data, read_offset, &arg);
}

#endif /* PACKET_DATA_H */
