/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
}

#include <libpandora/pandora.h>
#include <libpandora/stackfactory.h>
#include <libpandora/pandoraentry.h>
#include <libpandora/stackentry.h>
#include <libpandora/stackdesc.h>
#include <libpandora/dispatcher.h>
#include <libpandora/globalresourcemanager.h>
#include <libpandora/clock.h>
#include <libpandora/storage.h>
#include <libpandora/fileutil.h>
#include <libpandora/ltdl.h>
#include <iostream>

#if 0
#if defined(__GNUC__)
#if defined(__osf__) || defined (__FreeBSD__)
#include <map>
template class __default_alloc_template<1, 0>;
#endif

#if defined(__linux__)
template class __malloc_alloc_template<(int)0>;
#endif
#endif  /* __GNUC__ */
#endif

Pandora *pandora = NULL;

static bool pandora_update(const char *str, void *d)
{
  if (d == NULL) return false;
  Pandora *p = static_cast<Pandora *>(d);
  return p->init(str);
}

long Pandora::grm_id = string_hash("stacks");
stack_handle_t Pandora::curhandle = NIL_STACK_HANDLE;
Mutex Pandora::running_mx;

Pandora::Pandora(void) 
  : nbDisp(0), nbRunning(0) 
{
  if (pandora != NULL) {
    pandora_error("only one instance of Pandora may exist");
  } else {
    pandora = this;
  }
  if (grm != NULL) {
    grm->registerManager(grm_id, &pandora_update, this);
  }
} 

Pandora::~Pandora(void) 
{
  clean();

  pandora_assert(nbRunning == 0);

  StackEntry *se;
  valuesDo(repository, se) __DELETE(se);
  repository.clear();

  pandora = NULL;
}

bool Pandora::init(const char *defs) 
{
  if (defs == NULL) {
    pandora_warning("null stack definitions file");
    return false;
  }
  StackFactory sf(defs);
  StackEntry *se = NULL;
  while((se = sf.parseStack()) != NULL) {
    setStack(*se, false);
    __DELETE(se);
  }

  return true;
}

bool Pandora::update(void)
{
  if (grm == NULL) return false;
  return grm->update(grm_id);
}

stack_handle_t Pandora::start(const text &_id, bool threaded, bool root) 
{
  dclean();

  StackEntry *se = NULL;
  MultiValue values[MAX_OPTIONS];
  int nbvals = 0;
  text id;

  const char *vstart = strchr(_id.data(), '[');
  if (vstart != NULL) {
    nbvals = parseValues(vstart+1, values, MAX_OPTIONS);
    while(isspace(*--vstart));
    id.init(_id.data(), (vstart-_id.data()+1));
  } else {
    id = _id;
  }

  do {
    se = repository.atOrNil(id);
  } while (se == NULL && update());

  if (se == NULL) {
    pandora_notice("stack " << id << " is not defined");
    return NIL_STACK_HANDLE;
  }

  stack_handle_t handle = nextHandle();

  disp_t *d = new disp_t(handle, id, root);
  //pandora_debug("[new stack: " << id << " @" << (int) d->dispatcher << "]");
  if (!setup(d->dispatcher, se, values, nbvals)) {
    __DELETE(d);
    return NIL_STACK_HANDLE;
  }

  running_mx.lock();
  running.atPut(handle, d);
  running_mx.unlock();

  if (!root) d->dispatcher->depend();

  storage->save(Storage::state, d->id, (void *)handle);

  if (se->run) ++nbRunning;
  if (!d->dispatcher->start(threaded)) {
    pandora_notice("stack " << id << " cannot be started");
    finished(handle);
    return NIL_STACK_HANDLE;
  }

  return handle;
}

bool Pandora::stop(stack_handle_t sh) 
{
  dclean();

  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->stop();
}

stack_handle_t Pandora::stop(const text &id) 
{
  static Mutex mx;
  stack_handle_t sh = NIL_STACK_HANDLE;

  mx.lock();
  
  stack_handle_t handle;
  disp_t *d;
  keysValuesDo(running, handle, d) {
    if (d->id == id) {
      sh = handle;
      break;
    }
  }

  mx.unlock();

  if ((sh != NIL_STACK_HANDLE) && !stop (sh)) sh = NIL_STACK_HANDLE;
  return sh;
}

bool Pandora::stop_async(stack_handle_t sh) 
{
  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->stop_async();
}

bool Pandora::clean(stack_handle_t sh)
{
  dclean();

  bool status = false;
  disp_t *d = NULL;

  running_mx.lock();
  if (!checkHandle(sh)) 	goto failed;

  status = true;
  d = disp_entry(sh);
  running.removeKey(sh);

 failed:
  running_mx.unlock();
  
  if (status) {
    text id = d->id;
    __DELETE(d);
    --nbDisp;
    storage->save(Storage::state, id, (void *) -1);
  }

  return status;
}

bool Pandora::suspend(stack_handle_t sh) 
{
  dclean();
  
  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->suspend();
}


bool Pandora::resume(stack_handle_t sh) 
{
  dclean();

  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->resume();
}

bool Pandora::use(stack_handle_t sh) 
{
  if (!checkHandle(sh)) return false;

  //pandora_debug("[use #" << sh << "]");
  return dispatcher(sh)->use();
}

bool Pandora::release(stack_handle_t sh) 
{
  if (!checkHandle(sh)) return false;

  //pandora_debug("[release #" << sh << "]");
  return dispatcher(sh)->release();
}

stack_handle_t Pandora::use(const text &id) 
{
  static Mutex mx;
  stack_handle_t sh = NIL_STACK_HANDLE;
  bool exist = true;

  mx.lock();

  stack_handle_t handle;
  disp_t *d;
  keysValuesDo(running, handle, d) {
    if (d->id == id) {
      sh = handle;
      break;
    }
  }

  if (sh == NIL_STACK_HANDLE) {
    exist = false;
    sh = start(id, true, false);
  }

  mx.unlock();

  if (exist) use(sh);
  //pandora_debug("[use " << id << "]");
  return sh;
}

bool Pandora::push(stack_handle_t sh, Packet *pkt)
{
  bool ret = false;

  running_mx.lock();

  if (checkHandle(sh)) {
    ret = dispatcher(sh)->push(pkt);
  } else {
    pandora_notice("stack: @" << (long) sh << " does not exist");
    cleanPacket(pkt);
    ret = false;
  }

  running_mx.unlock();

  return ret;
}


StackEntry *Pandora::getStackEntry(const text &id)
{ 
  return repository.atOrNil(id);
}

int Pandora::listDefined(text *ids, int max_ids) 
{
  int i = 0;
  update();
  StackEntry *se;
  valuesDo(repository, se) {
    if (i < max_ids) {
      ids[i] = se->id;
      ++i;
    } else {
      goto out;
    }
  }
 out:
  return i;
}

int Pandora::listRunning(stack_handle_t *shs, int max_ids) 
{
  dclean();
  
  int i = 0;

  running_mx.lock();

  stack_handle_t handle;
  disp_t *d;
  keysValuesDo(running, handle, d) {
    if (i < max_ids && d->dispatcher->isRunning()) {
      shs[i] = handle;
      ++i;
    }
  }
  running_mx.unlock();

  return i;
}
  
bool Pandora::setStack(const StackEntry &se, bool dynupdate) 
{
  dclean();
  
  bool status = true;

  if (se.id.isNull()) {
    pandora_notice("cannot register stack with no name");
    return false;
  }

  StackEntry *new_se = new StackEntry(se);

  if (repository.includesKey(new_se->id)) {
    StackEntry *tmp = repository.at(new_se->id);
    repository.removeKey(new_se->id);
    __DELETE(tmp);

    if (dynupdate) {
      disp_t *d;
      running_mx.lock();
      valuesDo(running, d) {
	if (d->id == new_se->id) 
	  status = d->dispatcher->refresh(new_se) & status;
      }
      running_mx.unlock();
    }
  }

  repository.atPut(new_se->id, new_se);
  storage->save(Storage::stack, new_se->id, new_se);
  return status;
}


bool Pandora::setOption(const text &stk, const text &comp, const text &op, 
			const MultiValue &mv)
{
  dclean();
  
  if (stk.isNull()) return false;
  
  StackEntry *se = repository.atOrNil(stk);

  if (se == NULL) {
    pandora_notice("stack: " << stk << " does not exist");
    return false;
  }

  bool status = se->setOption(comp, op, mv);
  if (status) storage->save(Storage::stack, se->id, se);
  return status;
}

bool Pandora::setOption(stack_handle_t sh, const text &comp, const text &op, 
			const MultiValue &mv)
{
  dclean();
    
  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->setOption(comp, op, mv);
}

bool Pandora::getOption(stack_handle_t sh, const text &comp, const text &op, 
			MultiValue *mv) 
{
  dclean();
  
  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->getOption(comp, op, mv);
}

bool Pandora::query(stack_handle_t sh, const text &comp, const text &arg,
		    MultiValue *mv)
{
  dclean();

  if (!checkHandle(sh)) return false;
  return dispatcher(sh)->query(comp, arg, mv);
}

bool Pandora::expand(const text &sid, bool lup)
{
  StackEntry *se = NULL;
  do {
    se = repository.atOrNil(sid);
  } while(se == NULL && update());

  if (se == NULL) return false;
  return expand(se, lup);
}

bool Pandora::expand(StackEntry *se, bool lup)
{
  if (se == NULL) return false;

  int where = -1;
  while ((where = se->macro()) >= 0) {
    StackEntry *sub = repository.atOrNil(se->getCompEntry(where)->id);
    if (sub == NULL) {
      pandora_notice("cannot find specified substack");
      return false;
    }
    if (!se->insertStack(sub, where)) {
      pandora_notice("failed to insert substack");
      return false;
    }
  }

  if (lup) return se->lookup();
  return true;
}

bool Pandora::setup(Dispatcher *disp, StackEntry *se, 
		    MultiValue *vals, int nbvals)
{
  pandora_assert(disp != NULL);

  if (!expand(se, false)) return false;
  if (!disp->init(se, vals, nbvals)) return false;
  ++nbDisp;

  return true;
}

void Pandora::stop(void) 
{
  stack_handle_t handle;
  keysDo(running, handle) stop(handle);
}

void Pandora::stop_async(void) 
{
  stack_handle_t handle;
  disp_t *d;
  keysValuesDo(running, handle, d) {
    if (!d->root) continue;
    stop_async(handle);
  }
}

void Pandora::clean(void) 
{
  while (running.size() > 0) {
    stack_handle_t handle;
    keysDo(running, handle) clean(handle);
  }
}

void Pandora::clean_async(void)
{
  stack_handle_t handle;
  keysDo(running, handle) finished(handle);
}

void Pandora::finished(stack_handle_t sh) 
{
  //pandora_debug("adding stack #" << sh << " to clean list");
  toClean.put(sh);
}
  
void Pandora::dispCleanup(bool blocking) 
{
  if (!blocking && toClean.isEmpty()) return;
  do {
    stack_handle_t handle = toClean.get();
    //pandora_debug("removing stack #" << handle << " from clean list");
    if (!checkHandle(handle)) continue;
    if (clean(handle)) --nbRunning;
  } while (!toClean.isEmpty());
}

int Pandora::poll(bool blocking)
{
  if (blocking) {
    while (nbRunning > 0) dispCleanup(true);
  } else {
    dispCleanup(false);
  }

  return nbRunning;
}

int Pandora::poll(stack_handle_t sh, bool blocking)
{
  if (sh == NIL_STACK_HANDLE) return -1;

  if (blocking) {
    while (checkHandle(sh) && dispatcher(sh)->isRunning())
      dispCleanup(true);
  } else {
    dispCleanup(false);
  }  
  
  return (int) !(checkHandle(sh) && dispatcher(sh)->isRunning());
}

bool Pandora::getName(stack_handle_t sh, text &name)
{
  if (!checkHandle(sh)) return false;
  name = disp_entry(sh)->id;
  return true;
}

int Pandora::parseValues(const char *str, MultiValue *vals, int maxvals)
{
  int n = 0;

  MultiValue tmp;
  int ret;

  if (maxvals <= 0) return 0;
  StackFactory sf(str);

  while ((ret = sf.parseValue(&tmp)) >= 0) {
    if (ret) {
      vals[n] = tmp;
    } else {
      if (++n == maxvals) break;
    }
  }

  return n;
}


int pandora_init(void)
{
  char uri[4096];

  lt_dlinit();

  new GlobalResourceManager();
  if (grm == NULL) 		return -1;

  new DynLoader();
  if (dynloader == NULL) 	return -2;
  if (make_default_uri(uri, sizeof(uri), "libconfig", "PANDORA_LIBCONFIG"))
    grm->addResource(DynLoader::grm_id, uri, 900);

  new Pandora();
  if (pandora == NULL) 		return -3;
  if (make_default_uri(uri, sizeof(uri), "stacks", "PANDORA_STACKS"))
    grm->addResource(Pandora::grm_id, uri, 900);
  
  new Clock();
  if (wallclock == NULL)	return -4;

  
  new Storage();
  if (storage == NULL)		return -5;

  

  return 0;
}

int pandora_fini(void)
{
  delete pandora;
  if (pandora != NULL) 		return -3;
  delete dynloader;
  if (dynloader != NULL) 	return -2;
  delete grm;
  if (grm != NULL) 		return -1;
  lt_dlexit();

  delete storage;
  if (storage != NULL)		return -5;
  delete wallclock;
  if (wallclock != NULL)	return -4;

  return 0;
}

Pandora::disp_t::disp_t(stack_handle_t h, const text &i, bool r) 
  : dispatcher(new Dispatcher(h)), id(i), root(r) 
{
}

Pandora::disp_t::~disp_t(void) 
{ 
  __DELETE(dispatcher); 
}

void Pandora::dump(ostream *f)
{
  update();
  pandora_debug("will dump " << repository.size() << " entries");
  StackEntry *se;
  valuesDo(repository, se) {
    se->setup();
    se->print(f);
  }
}
