/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef SET_H
#define SET_H

#include <libpandora/global.h>

#include <libpandora/conf/string.h>
#include <libpandora/util.h>
#include <libpandora/error.h>

#define SET_DEBUG 0

// Generic set.  The element type must implement the following
// functions:
//
//	a hash function:		Elt::operator long() const
//	an equality function:		Elt::operator ==() const
//	an inequality function:		Elt::operator !=() const
//	an undefined element:		Elt::Elt()
//
// Note that the above functions work by default for primitive types.

#define GENERIC	template<class Elt> inline
#define SET	Set<Elt>

typedef long set_iterator_t;

template<class Elt>
class Set
{
public:
  long	 _tally;
  Elt	*_array;
  long	 _capacity;

  Elt	_undefined;

protected:
  static long sizeFor(long nElements);

public:
  Set(Elt undefined= Elt(), long nElements= 4);
  Set(const SET &other) {
    pandora_error("set copy");
    _array= 0;
    *this= other;
  }

  Set &operator=(const SET &other) { 
    pandora_error("set assignment " << (void *)&other);
  }

  virtual ~Set(void) { __DELETE_ARRAY(_array); }

public:
  bool includes(const Elt &anObject) const;
  void add(const Elt &newObject);
  void remove(const Elt &oldObject);
  long size(void) const { return _tally; }
  void grow(void);
  void rehash(void);
  void clear(void);
  void iterInit(set_iterator_t &iter) const { iter = -1; }
  Elt  &iterNextElement(set_iterator_t &iter);
  
protected:
  void newArray(long nElements);
  void atNewIndexPut(long index, const Elt &anObject);
  long  findElementOrNil(const Elt &anObject) const;
  void fixCollisionsFrom(long index);
  void fullCheck(void);
  long  growSize(void) const { return pandora_max(_capacity, 2); }
  void swap(long oneIndex, long otherIndex);

protected:
  virtual const long  scanFor(const Elt &anObject) const;
  virtual void	     noCheckAdd(const Elt &anObject);

public:
  // you never know when one of these will come in handy...
  inline static long stringHash(char *s) {
    long l, m;
    if ((l= m= strlen(s)) <= 2)
      if (l == 2) m= 3;
      else if (l == 1) return s[0] & 0x7f * 106;
      else return 21845;
    return s[0] * 48 + s[m - 2] + l;
  }
  // or even one of these...
  inline static long stringCompare(char *s, char *t) {
    return (!s && !t) || (s && t && !strcmp(s, t));
  }
};


#define elementsDo(SET, ELT) \
  (ELT)= (SET)._undefined; \
  for(long _0= 0; _0 < ((SET)._capacity); ++_0) \
    if (((ELT)= ((SET)._array[_0])) != (SET)._undefined) \

//
// INLINE IMPLEMENTATION
//

GENERIC SET::Set(Elt undefined, long nElements)
{
  _undefined= undefined;
  _tally= 0;
  newArray(sizeFor(nElements));
}


// Large enough size to hold nElements with some slop (see fullCheck).
// Ensure that the set is always a prime size to make probe more efficient.

GENERIC long SET::sizeFor(long nElements)
{
  return (nElements <= 0) ? 1 : ((nElements + 1) * 4) / 3;
}


// Initialize a new array.

GENERIC void SET::newArray(long nElements)
{
#if 1
  _capacity= 1;
  while (nElements) {
    _capacity<<= 1;
    nElements>>= 1;
  }
  _capacity-= 1;
#else
  _capacity= nElements;
#endif
  pandora_assert(_capacity >= nElements);
  _array= new Elt[_capacity];
  for (long index= 0; index < _capacity; ++index)
    _array[index]= _undefined;
  _tally= 0;
}


// Answer if the set includes anObject.

GENERIC bool SET::includes(const Elt &anObject) const
{
  pandora_assert(anObject != _undefined);
  return _array[findElementOrNil(anObject)] != _undefined;
}


// Add newObject to the set.

GENERIC void SET::add(const Elt &newObject)
{
  long index;
  pandora_assert(newObject != _undefined);
  index= findElementOrNil(newObject);
  if (_array[index] == _undefined) {
    atNewIndexPut(index, newObject);
  }
}


// Remove oldObject from the set.

GENERIC void SET::remove(const Elt &oldObject)
{
  long index;
  index= findElementOrNil(oldObject);
  pandora_assert(_array[index] != _undefined);
  _array[index]= _undefined;
  --_tally;
  fixCollisionsFrom(index);
}


// Add anObject to the set at index; assumes slot is empty.

GENERIC void SET::atNewIndexPut(long index, const Elt &anObject)
{
  _array[index]= anObject;
  ++_tally;
  fullCheck();
}


// Answer the index of a first slot containing either a nil
// (indicating an empty slot) or an element that matches the given
// object.  Answer the index of that slot.  Fail if neither a
// match nor an empty slot is found.

GENERIC long SET::findElementOrNil(const Elt &anObject) const
{
  long index;
  index= scanFor(anObject);
  // index==-1 => Bad scene.  Neither have we found a matching element
  // nor even an empty slot.  No hashed set is ever supposed to get
  // completely full.
  pandora_assert(index != -1);
  return index;
}


// The element at index has been removed and replaced by nil.
// This method moves forward from there, relocating any entries
// that had been placed below due to collisions with this one.

GENERIC void SET::fixCollisionsFrom(long index)
{
  long oldIndex, newIndex;
  oldIndex= index;

  while ((oldIndex= (oldIndex == _capacity - 1) ? 0 : oldIndex + 1),
	 (_array[oldIndex] != _undefined)) {
    newIndex= findElementOrNil(_array[oldIndex]);
    if (oldIndex != newIndex) swap(oldIndex, newIndex);
  }
}


// Check if the set is too full, grow if necessary.

GENERIC void SET::fullCheck()
{
  // Keep array at least 1/4 free for decent hash behavior.
  if (_capacity * 3 <= _tally * 4)
    grow();
}


// Grow the set.

GENERIC void SET::grow()
{
  // Grow the elements array and reinsert the old elements.
  const Elt *oldElements= _array;
  long oldCapacity= _capacity;
  //putchar('+'); fflush(stdout);
  newArray(oldCapacity + growSize());
  for (long i= 0; i < oldCapacity; ++i)
    if (oldElements[i] != _undefined)
      noCheckAdd(oldElements[i]);
  __DELETE_ARRAY(oldElements);
#if SET_DEBUG
  pandora_debug("set [" << this << "]: " 
		<< oldCapacity << " -> " << _capacity 
		<< " (" << _tally << " = " 
		<< (float)_tally / (float) _capacity << "%)");
#endif
}


// Add anObject to the set without checking for collision.

GENERIC void SET::noCheckAdd(const Elt &anObject)
{
  _array[findElementOrNil(anObject)]= anObject;
  ++_tally;
}


// An object was removed: rehash the contents of the set.

GENERIC void SET::rehash()
{
  // Grow the elements array and reinsert the old elements.
  const Elt *oldElements= _array;
  newArray(_capacity);
  for (long i= 0; i < _capacity; ++i)
    if (oldElements[i] != _undefined)
      noCheckAdd(oldElements[i]);
  __DELETE_ARRAY(oldElements);
}


// clear the set

GENERIC void SET::clear()
{
  for (long i= 0; i < _capacity; ++i)
    _array[i] = _undefined;
  _tally = 0;
}


// Scan the key array for the first slot containing either a nil
// (indicating an empty slot) or an element that matches
// anObject.  Answer the index of that slot or zero if no slot is
// found.  This method will be overridden in various subclasses that
// have different interpretations for matching elements.

GENERIC const long SET::scanFor(const Elt &anObject) const
{
  long start= (((unsigned long)anObject) * 2 % _capacity);

  // search from (hash mod size) to the end
  for (long index= start; index < _capacity; ++index)
    if ((_array[index] == _undefined) || (_array[index] == anObject))
      return index;

  // search from 0 to where we started
  for (long index= 0; index < start; ++index)
    if ((_array[index] == _undefined) || (_array[index] == anObject))
      return index;

  return -1;	// no match AND no empty slot
}


// Swap two elements in the set.
//
// NOTE: overridden by subclasses so that fixCollisions will work.

GENERIC void SET::swap(long oneIndex, long otherIndex)
{
  const Elt tmp= _array[oneIndex];
  _array[oneIndex]= _array[otherIndex];
  _array[otherIndex]= tmp;
}

GENERIC Elt &SET::iterNextElement(set_iterator_t &iter)
{
  while(++iter < _capacity)
    if (_array[iter] != _undefined) return _array[iter];
  return _undefined;
}

#undef GENERIC
#undef SET


#endif /* SET_H */
