/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <assert.h>
#include <libpandora/conf/string.h>
	   }

#include <libpandora/stackdesc.h>
#include <libpandora/compcache.h>
#include <libpandora/comprefstore.h>
#include <libpandora/timersupervisor.h>
#include <libpandora/stackentry.h>
#include <libpandora/component.h>
#include <libpandora/dynloader.h>
#include <libpandora/lcs.h>


#define REFRESH_DEBUG	0
#define CLEAN_DEBUG	0
#define CREATE_DEBUG	0

StackDesc::StackDesc(void) 
  : cache(NULL), store(NULL), supervisor(NULL),
    compList(NULL), nbComps(0),
    handle(NIL_STACK_HANDLE), run(true)
{
  pandora_assert(dynloader != NULL);
}

StackDesc::~StackDesc(void) 
{ 
  __DELETE(supervisor);
  __DELETE(store);
  __DELETE(cache);// cache *must* be flushed before stack cleanup
  cleanup(); 
}

bool StackDesc::init(stack_handle_t h, StackEntry *se, 
		     MultiValue *vals, int nbvals) 
{
  __DELETE(supervisor);
  __DELETE(store);
  if (cache != NULL) {
    __DELETE(cache);
    cleanup();
  }

  handle = h;
  id = se->id; 
  run = se->run;
  se->setup();

  if (!initComps(se, vals, nbvals)) return false;
  if (nbComps == 0) return false;

  cache = 	new CompCache(nbComps);
  store = 	new CompRefStore(nbComps);
  supervisor = 	new TimerSupervisor();

  //se->print();

  return true;
}
  
bool StackDesc::initComps(StackEntry *se, MultiValue *vals, int nbvals)
{
  int n = se->getNbComps();
  if (n <= 0) return false;

  stack_env_t env;
  if (vals != NULL) {
    int nvars = pandora_min(se->getNbParams(), nbvals);
    for (int i = 0; i < nvars; ++i)
      env.atPut(se->getParam(i), vals[i]);
  }

  compList = new CompEntry[n];

  for (int i = 0; i < n; ++i) {
    compList[nbComps] = *(se->getCompEntry(i));
    CompEntry *ce = &(compList[nbComps]);
    ce->setVars(&env);
    if (ce->isParam() && (ce->alias == ce->id)) continue;
    if (!ce->setup()) {
      pandora_warning("cannot create: " << ce->id);
      return false;
    }
    ++nbComps;
  }

  setStaticOptions();

  //if (check()) pandora_notice("stack " << id << ": check passed");

  return true;
}

void StackDesc::setStaticOptions(void)
{
  for (int i = 0; i < nbComps; ++i) {
    bool store = false, statics = false;
    CompEntry *ce = getCompEntry(i);
    Component *comp = staticComps.atOrNil(i+1);
    if (comp == NULL) {
      if (!ce->isComponent()) continue;
      if (!ce->getComponent(&comp)) continue;
      comp->ind = i;
      store = true;
    }
      
    for (int j = 0; j < ce->nbOptions; ++j) {
      OptionEntry *oe = ce->getOptionEntry(j);
      statics = comp->setStaticOption(oe) | statics;
    }
    if (store) {
      if (statics) staticComps.atPut(i+1, comp);
      else __DELETE(comp);
    } else {
      if (!statics) {
	staticComps.removeKey(i);
	__DELETE(comp);
      }
    }
  }
}

bool StackDesc::check(void)
{
  return _check(0, (symbol_id_t) -1);
}

bool StackDesc::_check(int index, symbol_id_t type)
{
  if (index == nbComps) {
    if (type != 0) {
      pandora_warning("not terminated stack");
      return false;
    } else {
      return true;
    }
  }

  symbol_id_t ids[16];
  int n = 0;

  CompEntry *ce = getCompEntry(index);
  if (ce->needLookup) {
    pandora_warning("cannot lookup component at position " << index);
    return false;
  }

#if 0
  if (index > 0) 
    pandora_debug("packet type:        " << Packet::get_name(type));
  pandora_debug("checking component: " << ce->id << " at pos " << index);
#endif

  symbol_id_t cid = DynLoader::make_id(ce->id);
  
  bool ok = false;

  if (type == 0) {
    pandora_warning("stack " << id << ": component " << ce->id 
		    << " cannot be reached"); 
    return false;
  } else if (type == (symbol_id_t) -1) {
    ok = true;
  } else {
    n = Component::pkt_in.atOrNil(cid, ids, sizeof(ids)/sizeof(symbol_id_t));
    for (int i = 0; i < n; ++i) {
      if ((ids[i] == (symbol_id_t) -1)
	  || (Packet::extends(type, ids[i]))) {
	ok = true;
	break;
      }
    }
  }

  if (!ok) {
    pandora_warning("stack " << id << ": component " << ce->id 
		    << " does not handle packets of type: "
		    << Packet::get_name(type));
    return false;
  }

  n = Component::pkt_out.atOrNil(cid, ids, sizeof(ids)/sizeof(symbol_id_t));
  if (n == 0) return _check(index + 1, 0);

  for (int i = 0; i < n; ++i) {
    symbol_id_t ntype = ((ids[i] == (symbol_id_t) -1) ? type : ids[i]);
    bool ok0 = false;
    if (ce->branches > 0) {
      return true;
      for (int b = 0; b < ce->branches; ++b) {
      }
    } else {
      ok0 = _check(index + 1, ntype);
    }

    if (ok0) return true;
  }

  return false;
}

void StackDesc::setNextComponent(Component *calling, Component **ptr, 
				  int branch)
{

  pandora_assert(calling != NULL);
  pandora_assert(ptr != NULL);

  int prevIndex = calling->ind;

#if CREATE_DEBUG
  pandora_debug("next comp for #" << prevIndex << " [" << branch << "]");
#endif

  if (*ptr == NULL) {
    CompEntry *pce = getCompEntry(prevIndex);

    if (branch > pce->branches) return;

    int compIndex = prevIndex 
      + ((branch == 0) 
	 ? pce->mux
	 : ((branch > 0) ? pce->sw[branch] : 1));

    if (compIndex == prevIndex) {
      if (calling->prevDemux == NULL) return;
      setNextComponent(calling->prevDemux, 
		       calling->prevDemux->refdComponent, 0);
      return;
    }

    if ((compIndex < 0) | (compIndex > nbComps)) return;

    Component *prevDemux = (branch != 0) ? calling : calling->prevDemux;

    initComponent(ptr, compIndex, prevDemux);
    if (*ptr == NULL) return;
  } else {
    pandora_info("comp #" << (*ptr)->ind << " already set");
    (*ptr)->referrer = ptr;
    (*ptr)->refer();
  }

  (*ptr)->prevComponent = calling;
  (*ptr)->branch = ((branch != 0) ? branch : calling->branch);
  (*ptr)->demuxed = (branch != 0);

#if CREATE_DEBUG
  pandora_debug("created component: " << *ptr << " @" << (*ptr)->ind
		<< " [" << (*ptr)->branch << "]");
#endif
}


void StackDesc::setFirstComponent(Component **ptr)
{
  initComponent(ptr, 0, NULL);
}

void StackDesc::initComponent(Component **ptr, int index,
			      Component *prevDemux) 
{
  if (index >= nbComps) {
    pandora_error("trying to push to inexistant component");
  }
  CompEntry *ce = getCompEntry(index);
  Component *component = getComponent(index);
  pandora_assert( component != NULL );
  component->compStack = this;
  component->timerSupervisor = supervisor;
  component->referrer = ptr;
  component->refer();

  if (ce->nbOptions > 0)
    component->setOptions(ce->optionList, ce->nbOptions);

  component->prevDemux = prevDemux;
  
  component->refdComponent = (ce->mux != 0 | prevDemux == NULL) 
    ? &(component->nextComponent)
    : prevDemux->refdComponent;

  component->setup();

  pandora_assert(ptr != NULL);
  *ptr = component;
}

void StackDesc::clean(Component *component)
{
  if (component == NULL) return;

  do {
    Component *ctmp = NULL;
    pandora_assert(component->compStack == this);
    component->release();
#if CLEAN_DEBUG
    pandora_debug("cleaning   comp [" << component << "] @" << component->ind 
		 << " (" << component->refCount << ")");
#endif

    if (component->inUse() || component->flushed) break;
    component->flushed = true;
    Component **ref = component->referrer;
    if (ref != NULL) *ref = NULL;

#if CLEAN_DEBUG
    pandora_debug("removing   comp [" << component << "] @" << component->ind
		 << " (" << component->refCount << ")");
#endif

    component->_cleanup();

    ctmp = component->nextComponent;
    if (ctmp != NULL 
	&& ctmp->referrer == component->refdComponent)
      ctmp->referrer = NULL;
    component->nextComponent = NULL;
    releaseComponent(component);
    component = ctmp;
  } while (component != NULL);
}

void StackDesc::cleanup(void) 
{
  Component *sComp;
  valuesDo(staticComps, sComp) __DELETE(sComp);
  staticComps.clear();

  __DELETE_ARRAY(compList);
  handle = NIL_STACK_HANDLE;
  nbComps = 0;
}


#if 0
bool StackDesc::refreshBinding(const text &id)
{
  pandora_warning("not implemented");
  return false;
}

bool StackDesc::refreshBindings(void)
{
  bool status = true;
  if (cache != NULL) cache->flush();

  for (int i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    void *func = dynloader->refresh((symbol_id_t)ce->id, (void *)ce->func);
    status = status & (func != NULL);
    if (func != NULL) ce->func = (Component *(*)(void)) func;
  }

  return status;
}
#endif

bool StackDesc::setOption(const text &_comp, const text &op, 
			  const MultiValue &mv)
{
  bool exist = false, status = true;
  if (_comp.isNull()) return false;

  text comp;
  if (!dynloader->lookup(comp, _comp)) return false;

  int i = nbComps;

  for (i = 0; i < nbComps; ++i) {
    if (getCompId(i) == comp) {
      exist = true;
      CompEntry *ce = getCompEntry(i);
      status = ce->setOption(op, mv) & status;
      
      if (ce->nbOptions > 0) {
	Component *c;
	componentsDo(*store, i, c) {
	  c->setOptions(ce->optionList, ce->nbOptions);  
	  c->setup();
	}
      }
    }
  }

  return (exist & status);
}

bool StackDesc::getOption(const text &cname, const text &op, MultiValue *mv)
{
  if (cname.isNull()) 		return false;
  if (mv == NULL) 		return false;

  text cid;
  if (!dynloader->lookup(cid, cname)) 	return false;

  for (int i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    if (ce->id != cid) continue;

    Component *target = NULL;
    componentsDo(*store, i, target) break;

    if (target != NULL) {
      text orig;
      return target->getOption((ce->getOrigName(orig, op) ? orig : op), mv);
    }
  }

  pandora_warning("component " << cid << " not found");
  return false;
}

bool StackDesc::query(const text &cname, const text &arg, MultiValue *mv)
{
  if (cname.isNull()) 		return false;
  if (mv == NULL) 		return false;

  text cid;
  if (!dynloader->lookup(cid, cname)) 	return false;

  for (int i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    if (ce->id != cid) continue;

    Component *target = NULL;
    componentsDo(*store, i, target) break;

    if (target != NULL) return target->query(arg, mv);
  }

  pandora_warning("component " << cid << " not found");
  return false;
}

static bool ce_equal(const CompEntry &a, const CompEntry &b) {
  return (a.id == b.id);
}

struct refresh_t {
  StackDesc *oldSD, *newSD;
  lcs_op_t *oldOps, *newOps;
};

void StackDesc::collapse(Component *comp, void *r)
{
  refresh_t *ref = (refresh_t *)r;

  pandora_assert(comp != NULL);

  int i = comp->ind;
  switch ((ref->oldOps[i]).type) {

  case lcs_op_t::lcs_match: {
    int dst = (ref->oldOps[i]).dst;
    ref->oldSD->store->remove(comp);
    comp->ind =             dst;
    CompEntry *ce =         ref->newSD->getCompEntry(dst);
    comp->cacheable =       false;
    comp->compStack =       ref->newSD;
    //comp->timerSupervisor = ref->newSD->supervisor;
    if (ce->nbOptions > 0) 
      comp->setOptions(ce->optionList, ce->nbOptions);
    comp->refdComponent = (ce->mux != 0 || comp->prevDemux == NULL) 
      ? &(comp->nextComponent)
      : comp->prevDemux->refdComponent;
    ref->newSD->store->add(comp);
  } break;

  case lcs_op_t::lcs_del: {
    comp->cleanBranches(); // it is meaningless to keep them

    if (comp->referrer != NULL) {
      *(comp->referrer) = comp->nextComponent;
    }
    
    if (comp->nextComponent != NULL) {
      comp->nextComponent->referrer = comp->referrer;
      comp->nextComponent->prevComponent = comp->prevComponent;
      comp->nextComponent->prevDemux = comp->prevDemux;
      comp->nextComponent->refer();
    } else if (comp->prevComponent != NULL) {
      if (comp->demuxed
	  /* && comp->prevComponent->branch != comp->branch */) {
	comp->prevComponent->closeBranch(comp->branch);
      }
      if (comp->prevDemux != NULL
	  && comp->refdComponent == comp->prevDemux->refdComponent
	  && comp->prevComponent->prevDemux != NULL) {
	comp->prevComponent->refdComponent = 
	  comp->prevComponent->prevDemux->refdComponent;
      }
    }
    comp->referrer = NULL;
    comp->prevComponent = NULL;
    comp->prevDemux = NULL;
  }  break;

  default: 
    pandora_error("unsupported operation"); 
    break;
  }
}

void StackDesc::insertComponent(Component *calling, Component **ptr, 
				int branch)
{
  pandora_assert(calling != NULL);
  pandora_assert(ptr != NULL);
  
  pandora_debug("inserting component after #" << calling->ind);

  Component *p = (*ptr);
  *ptr = NULL;
    
  setNextComponent(calling, ptr, branch);
  
  pandora_assert(*ptr != NULL);
  (*ptr)->nextComponent = p;
  if (p != NULL) { 
    p->referrer = ptr;
    p->prevComponent = *ptr;
    p->prevDemux = (*ptr)->prevDemux;
  }
}

void StackDesc::expand(Component *comp, void *r)
{
  refresh_t *ref = (refresh_t *)r;

  pandora_assert(comp != NULL);

  int i = comp->ind;
  CompEntry *ce = ref->newSD->getCompEntry(i);

  if (comp->nbBranches() > 0) {
    set_iterator_t iter;
    comp->nextComponents.iterInit(iter);
    Component **bcomp = NULL;
    while((bcomp = comp->nextComponents.iterNextValue(iter)) != NULL) {
      if (*bcomp == NULL) continue;
      int branch = (*bcomp)->branch;
      int k0 = i + ((branch > 0) ? ce->sw[branch] : 1);
      int k = (*bcomp)->ind;

      if (k != k0 && k != i) 
	ref->newSD->insertComponent(comp, bcomp, branch);
    } 
  }

  int j0 = i + ce->mux;
  int j = (comp->nextComponent != NULL) 
    ? comp->nextComponent->ind
    : i;
    
  if (j != j0 && j != i) 
    ref->newSD->insertComponent(comp, comp->refdComponent, 0);
}

void StackDesc::printStack(void)
{
  Component *comp = NULL;
  componentsDo(*store, 0, comp) {
    Component::apply(comp, &Component::print, NULL);
  }
}

bool StackDesc::refresh(StackDesc *old)
{
#if REFRESH_DEBUG
  pandora_debug("[stack refresh]");
#endif

  CompCache    *newCache = cache, *oldCache = old->cache;
  CompRefStore *newStore = store, *oldStore = old->store;

  if (oldStore == NULL) {
    for (int i = 0; i < old->nbComps; ++i) old->cleanStore(i);
    return false;
  }

  lcs_t<CompEntry> new_desc, old_desc;
  new_desc.elts = compList;
  new_desc.nelt = nbComps;
  old_desc.elts = old->compList;
  old_desc.nelt = old->nbComps;
  lcs_script_t script;

  lcs_print(&old_desc, &new_desc, &ce_equal, &script);
  bool ret = ((script.nops > 0) 
	      && (script.ops[0].type == lcs_op_t::lcs_match));

#if REFRESH_DEBUG
  script.print();  
#endif

  // XXX we don't want to lose previously set timeouts, ugly! :(
  __DELETE(supervisor);
  supervisor = old->supervisor;
  old->supervisor = NULL;

  refresh_t ref;

  ref.oldSD = old;
  ref.newSD = this;

  // ops rewriting
  ref.oldOps = new lcs_op_t[old->nbComps];
  ref.newOps = new lcs_op_t[nbComps];
  for (int i = 0; i < script.nops; ++i) {
   lcs_op_t *op = &(script.ops[i]);
   if (op->src >= 0) ref.oldOps[op->src] = (*op);
   if (op->dst >= 0) ref.newOps[op->dst] = (*op);   
  }

  oldCache->setFlushAll();

#if REFRESH_DEBUG
  pandora_debug("[original state]");
  old->printStack();
#endif

  // old stack collapse
  Component *root = NULL;
  componentsDo(*oldStore, 0, root) {
    Component::apply(root, collapse, (void *)&ref);
  }

#if REFRESH_DEBUG
  pandora_debug("[after collapse]");
  printStack();
#endif
  
#if REFRESH_DEBUG
  pandora_debug("[clean old cache]");
#endif
  // old stack clean
  for (int i = 0; i < old->nbComps; ++i) old->cleanStore(i);
  
#if REFRESH_DEBUG
  pandora_debug("[static options move]");
#endif
  Component *statComp = NULL;
  int si;
  keysValuesDo(old->staticComps, si, statComp) {
    int src = statComp->ind;
    int dst = ref.oldOps[src].dst;
    if (dst < 0) __DELETE(statComp);
    else staticComps.atPut(si + nbComps, statComp);
  }
  (old->staticComps).clear();
  
  if (newStore != NULL) {
#if REFRESH_DEBUG
    pandora_debug("[after clean]");
    printStack();
#endif
    
    // new stack expand
    root = NULL;
    componentsDo(*newStore, 0, root) {
      Component::apply(root, expand, (void *)&ref);
    }
    
#if REFRESH_DEBUG
    pandora_debug("[after insert]");
    printStack();
#endif
  }

#if REFRESH_DEBUG
  pandora_debug("[flush old cache]");
#endif
  oldCache->flush();

  __DELETE_ARRAY(ref.oldOps);
  __DELETE_ARRAY(ref.newOps);

  return ret;
}

void StackDesc::cleanStore(int index) 
{
  CompRefStore::CompSet *comp_set = &(store->store[index]);
  while(comp_set->size() > 0) {
    Component *comp = NULL;
    elementsDo(*comp_set, comp) {
#if REFRESH_DEBUG
      pandora_debug("cleaning   comp [" << comp << "] #" << index);
#endif
      Component::clean(comp);
    }
  }
}
