/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef STACK_DESC_H
#define STACK_DESC_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
	   }

#include <libpandora/error.h>
#include <libpandora/compentry.h>
#include <libpandora/optionentry.h>
#include <libpandora/compcache.h>
#include <libpandora/packetfactory.h>
#include <libpandora/comprefstore.h>
#include <libpandora/map.h>

class StackEntry;
class OptionEntry;
class Component;
class TimerSupervisor;

class StackDesc {
private:
  PacketFactory		pf;
  CompCache 		*cache;
  CompRefStore 		*store;
  TimerSupervisor 	*supervisor;

  CompEntry 		*compList;
  Map<int, Component *> staticComps;
  int 			nbComps;

  stack_handle_t	handle;
  text 			id;
  bool 			run;
  
public:
  StackDesc(void);
  ~StackDesc(void);

  void cleanup(void);

  void setNextComponent(Component *calling, Component **ptr, int branch);
  void setFirstComponent(Component **ptr);
  void clean(Component *);

  inline void setCacheFlush(void) ;
  inline Component *getComponent(int index);
  inline void releaseComponent(Component *comp);

  inline void usePacketType(Packet *);

  text getCompId(int index) const	{ return (getCompEntry(index))->id; }
  int getMux(int index) const		{ return (getCompEntry(index))->mux; }
  int getSize(void) const		{ return nbComps; }
  
  bool init(stack_handle_t h, StackEntry *se, MultiValue *vals, int nbvals);
  bool check(void);

  bool refresh(StackDesc *);
#if 0
  bool refreshBinding(const text &id);
  bool refreshBindings(void);
#endif

  inline stack_handle_t getHandle(void) { return handle; }
  inline const text &getID(void) const  { return id; }
  inline bool isRunable(void) const     { return run; }

private:
  void initComponent(Component **, int, Component *);
  void adjustComponent(Component *);

  void setStaticOptions(void);
  bool _check(int, symbol_id_t);

  bool initComps(StackEntry *se, MultiValue *vals, int nbvals);

  static void collapse(Component *, void *);
  static void expand(Component *, void *);
  void insertComponent(Component *, Component **, int);

  void cleanStore(int index);
  void printStack(void);

  inline CompEntry *getCompEntry(int index) const;

  bool setOption(const text &comp, const text &op, const MultiValue &mv);
  bool getOption(const text &cname, const text &arg, MultiValue *mv);
  bool query(const text &cname, const text &oname, MultiValue *mv);

  friend class Dispatcher;
};


void StackDesc::setCacheFlush(void) 
{
  pandora_assert(cache != NULL);
  cache->setFlushAll();
}

Component *StackDesc::getComponent(int index) 
{
  pandora_assert(cache != NULL && store != NULL);
  Component *comp = cache->get(index, getCompEntry(index));
  if (comp != NULL) store->add(comp);
  return comp;
}

void StackDesc::releaseComponent(Component *comp) 
{
  pandora_assert(cache != NULL && store != NULL);
  store->remove(comp);
  cache->release(comp);
}

CompEntry *StackDesc::getCompEntry(int index) const
{
  pandora_assert((index >=0) & (index < nbComps));
  return &(compList[index]);
}

void StackDesc::usePacketType(Packet *pkt)
{
  pf.load(pkt->getID());
}

#endif /* STACK_DESC_H */


