/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iomanip>

#include <libpandora/stackentry.h>
#include <libpandora/dynloader.h>
#include <libpandora/map.h>
#include <libpandora/serialize.h>

#define DEBUG_STACKENTRY 0

StackEntry::StackEntry(const StackEntry &x) : 
  needSetup(true), nbComps(x.nbComps), nbParams(x.nbParams),
  id(x.id), run(x.run) 
{
  for (int i = 0; i < nbComps; ++i)
    compList[i] = x.compList[i];
  for (int i = 0; i < nbParams; ++i)
    paramList[i] = x.paramList[i];
}

StackEntry &StackEntry::operator=(const StackEntry &x) 
{
  needSetup = true; nbComps = x.nbComps; nbParams = x.nbParams;
  id = x.id; run = x.run; 
  for (int i = 0; i < nbComps; ++i)
    compList[i] = x.compList[i];
  for (int i = 0; i < nbParams; ++i)
    paramList[i] = x.paramList[i];
  return *this;
}
  

size_t StackEntry::write(char *str, size_t maxlen) const
{
  size_t count = 0;

  serialVar(id);
  serialVar(run);

  serialVar(nbComps);
  for (int i = 0; i < nbComps; ++i) 
    serialVar(compList[i]);

  serialVar(nbParams);
  for (int i = 0; i < nbParams; ++i)
    serialVar(paramList[i]);

  return count;
}

size_t StackEntry::read(const char *str)
{
  size_t count = 0;

  needSetup = true;

  unserialVar(id);
  unserialVar(run);

  unserialVar(nbComps);
  for (int i = 0; i < nbComps; ++i)
    unserialVar(compList[i]);

  unserialVar(nbParams);
  for (int i = 0; i < nbParams; ++i)
    unserialVar(paramList[i]);

  return count;
}


bool StackEntry::setup(void)
{
  if (nbComps == 0) {
    pandora_warning("empty stack");
    return false;
  }
  if (!run) return false;
  Map<text, int> ranks;
  
  int order = 0, depth = 0;
  for (int i = 0; i < nbComps; ++i) {
    CompEntry *ce = &compList[i];
    
    // rank
    ce->rank = ranks.atOrNil(ce->id);
    ranks.atPut(ce->id, (ce->rank + 1));
    
    // order & depth
    ce->order = order;
    ce->depth = depth;
    if (ce->type & COMP_DEMUX) 	order += GET_COMP_DEMUX_FACT(ce->type);
    if (ce->type & COMP_PREMUX) order -= GET_COMP_DEMUX_FACT(ce->type);
    if (ce->type & COMP_SWITCH) depth += GET_COMP_SWITCH_FACT(ce->type);
    if (ce->type & COMP_LAST)	depth -= GET_COMP_SWITCH_FACT(ce->type);
  }

  setBranches();
  setMux();
  setSW();

  needSetup = false;

  return true;
}

void StackEntry::print(ostream *f)
{
  *f << '%' << id ;
  if (nbParams > 0) {
    *f << '(';
    for (int i = 0; i < nbParams; ++i) {
      *f << paramList[i];
      if (i < (nbParams-1)) *f << ", ";
    }
    *f << ')';
  }
  *f << " { ";
  for (int i = 0; i < nbComps; ++i) {
#if DEBUG_STACKENTRY 
    *f << "\n" << setw(2) << setfill(' ') << i << ": ";
#endif
    compList[i].print(f);
  }
#if DEBUG_STACKENTRY 
  *f << "\n";
#endif
  *f << "}\n";
}


void StackEntry::_print_graph(ostream *f, int idx, int b, int mux, int bmux) 
{
  if (idx >= nbComps-1) return;
  CompEntry *ce = getCompEntry(idx);
  int type = ce->type;

  int mux2 = (ce->mux > 0 ? idx+ce->mux : mux);
  int bmux2 = (ce->mux > 0 ? b : bmux);

#if 0
  pandora_debug("idx=" << idx << ", " << "mux=" << mux << ", "
		<< "b=" << b << ", " << "bmux=" << bmux);
#endif

#define isTypeOf(t, m)	(((t) & (m)) == (m))

#define edge(i2, p2)				\
  *f << "\""						\
  << getCompEntry((idx))->id << '|' << b << '|' << idx	\
  << "\" -> \""						\
  << getCompEntry((i2))->id << '|' << p2 << '|' << i2	\
  << "\";\n"
  
  *f << "\"" << getCompEntry((idx))->id << '|' << b << '|' << idx 
     << "\" [label=\"" << ce->id << "\",shape=ellipse,style=\"";
#if 1
  if (isTypeOf(type, COMP_DEMUX)) {
    *f << "filled";
  } else if (isTypeOf(type, COMP_SWITCH)) {
    *f << "filled";
  } else {
    *f << "solid";
  }
#else
    *f << "solid";
#endif
  *f << "\"];\n";

  if (isTypeOf(type, COMP_DEMUX)) {
    //pandora_debug("[demux]");

    for (int k = 1; k < 3; ++k) {
      int k2 = 10*b + k;
      edge(idx+1,k2);
      _print_graph(f, idx+1, k2, mux2, bmux2);
    }

    if (ce->mux == 0) return;
    _print_graph(f, idx+ce->mux, b, mux, bmux);
    return;
  } 
  
  if (isTypeOf(type, COMP_PREMUX)) {
    //pandora_debug("[premux]");    
    edge(mux, bmux);
    return;    
  }

  if (isTypeOf(type, COMP_SWITCH)) {
    for (int i = 1; i < ce->branches; ++i) {
      //pandora_debug("[switch] #" << i);
      edge(idx+ce->sw[i], b);
      _print_graph(f, idx+ce->sw[i], b, idx+ce->mux, bmux);
    }
    
    //pandora_debug("[switch] default");
    edge(idx+ce->mux, b);

    if (ce->mux == 0) return;
    _print_graph(f, idx+ce->mux, b, mux, bmux);
    return;
  }

  if (isTypeOf(type, COMP_LAST)) {
    //pandora_debug("[last]");    
    edge(mux, b);
    return;
  } 

  if (isTypeOf(type, COMP_INTER)) {
    //pandora_debug("[inter]");    
    edge(mux, b);
    return;
  } 

  //pandora_debug("[simple]");    
  edge(idx+1, b);
  _print_graph(f, idx+1, b, mux, bmux);
}

void StackEntry::print_graph(ostream *f)
{
  *f << "digraph " << id << " {\n\n";
  _print_graph(f, 0, 0, -1, 0);
  *f << "\"" 
     << getCompEntry(nbComps - 1)->id << "|0|" << nbComps - 1 
     << "\" [label=\"" << getCompEntry(nbComps - 1)->id 
     << "\",shape=ellipse];\n";
    
  *f << "}\n";
}

bool StackEntry::includes(const text &cid)
{
  return getCompEntry(cid) != NULL;
}

bool StackEntry::insertStack(StackEntry *se, int where)
{
  pandora_assert(se != NULL);
  pandora_assert(where >= 0 && where < nbComps);
  int inc = se->getNbComps();

  if (nbComps + inc > MAX_COMPONENTS) return false;

  CompEntry *macro = &(compList[where]);
  int typ = ((macro->type) & (~COMP_MACRO));

  stack_env_t env;
  int nvars = pandora_min(se->getNbParams(), macro->getNbOptions());
  for (int i = 0; i < nvars; ++i)
    env.atPut(se->getParam(i), macro->getOptionEntry(i)->mv);

  if (inc > 1) {
    for (int i = (nbComps-1); i > where; --i)
      compList[i+inc-1] = compList[i];
  }

  for (int i = 0; i < inc; ++i) {
    compList[where+i] = *(se->getCompEntry(i));
    (compList[where+i]).setVars(&env);
  }

  compList[where+inc-1].type |= typ;

  nbComps += (inc-1);

  needSetup = true;
  return true;
}

int StackEntry::macro(void)
{
  for (int i = 0; i < nbComps; ++i)
    if ((compList[i].type & COMP_MACRO) == COMP_MACRO)
      return i;
  return -1;
}

bool StackEntry::setOption(const text &_comp, const text &op, 
			   const MultiValue &val)
{
  if (_comp.isNull()) return false;

  text comp;
  if (!dynloader->lookup(comp, _comp)) return false;

  bool status = true, exists = false;
  
  for (int i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    ce->lookup();
    if (ce->id == comp) {
      exists = true;
      status = ce->setOption(op, val) & status;
    }
  }
  return status & exists;
}

bool StackEntry::getOrigName(text &orig, const text &alias)
{
  CompEntry *ce = getCompEntry(alias);
  if (ce == NULL) 		return false;
  if (ce->alias != alias) 	return false;
  orig = ce->id;
  return true;
}

void StackEntry::setBranches(void) {
  int index = 0, bid = 0;
  _setBranches(0, &index, &bid);
}

void StackEntry::_setBranches(int depth, int *index, int *bid)
{
  int b = 0;
  int mybid = *bid;
  CompEntry *ce = NULL; 
  for(/*empty*/; *index < nbComps; ++(*index)) {
    ce = getCompEntry(*index);
#if 0
    pandora_debug(*index << " " << mybid << " " 
	 << depth << " " << ce->depth << " " << b);
#endif
    if (ce->depth > 0) {
      if (ce->depth > depth) {
	++(*bid);
	_setBranches(ce->depth, index, bid);
      } else { 
	ce = getCompEntry(*index);
	ce->branch = b;
	if (GET_COMP_TYPE(ce->type) & COMP_INTER) ++b;
	if (GET_COMP_TYPE(ce->type) & COMP_LAST) return;
      }
    }
  }
}

void StackEntry::setMux(void)
{
  int i, j;

  for (i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    if (GET_COMP_TYPE(ce->type) & COMP_DEMUX) {
      int order = ce->order;
      int depth = ce->depth;
      int branch = ce->branch;
      for (j = i+1; 
	   (j < nbComps) 
	     && (getCompEntry(j))->order > order
	     && (getCompEntry(j))->depth >= depth; 
	   ++j) /* empty */;
      //pandora_debug(i << " / " << j);

      ce->mux = 0;
      if (j < nbComps) {
	CompEntry *ce2 = getCompEntry(j);
	if (ce2->order == order 
	    && ce2->depth == depth
	    && ce2->branch == branch) {
	  ce->mux = (j - i);
	  getCompEntry(j-1)->mux = 0;
	}
      } 
    }
  }
  CompEntry *last = getCompEntry(nbComps - 1);
  last->mux = 0;
}

void StackEntry::setSW(void)
{
  int i, j, k;

  for (i = 0; i < nbComps; ++i) {
    CompEntry *ce = getCompEntry(i);
    if (GET_COMP_TYPE(ce->type) & COMP_SWITCH) {
      int depth = ce->depth;
      k = 0;
      for (j = i+1; 
	   j < nbComps && getCompEntry(j)->depth > depth; 
	   ++j) {
	CompEntry *ce2 = getCompEntry(j-1);
	if ((GET_COMP_TYPE(ce2->type) & COMP_INTER)
            && (ce2->depth == depth+1)) {
	  ce->sw[k+2] = (j - i);
	  ce2->mux = 0;
	  //pandora_debug("branch #" << k+2 << ": " << i << " -> " << j);
	  ++k;
	}
      }

      CompEntry *celast = getCompEntry(j-1);
      if ((GET_COMP_TYPE(celast->type) & COMP_LAST)
          && (celast->depth == depth+1)) {
	ce->sw[0] = 0;
	if (j < nbComps) {
	  CompEntry *ce2 = getCompEntry(j);
	  if (ce2->depth == depth
	      && ce2->order == ce->order 
	      && ce2->branch == ce->branch) {
	    ce->sw[0] = (j - i);
	    celast->mux = 0;
	  }
	}
	//pandora_debug("branch #0: " << i << " -> " << j);
	++k;
      }
      ce->branches = k+1;
      ce->mux = ce->sw[0];
      ce->sw[1] = 1;
    }
  }
}
