/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <signal.h>
#include <libpandora/conf/snprintf.h>
#include <libpandora/conf/unistd.h>
}

#include <strstream>
#include <libpandora/storage.h>
#include <libpandora/fileutil.h>

#include <libpandora/stackentry.h>
#include <libpandora/dynlibrary.h>
#include <libpandora/dynsymbol.h>

Storage *storage = NULL;

Storage::Storage(void)
{
  *dir = '\0';
  if (storage != NULL) {
    pandora_error("only one instance of Storage may exist");
  } else {
    storage = this;
  }
}

Storage::~Storage(void)
{
  storage = NULL;
}

bool Storage::init(const char *name)
{

  const char *pdir = pandora_dir();
  if (pdir == NULL) return false;

  if (name == NULL) {
    int pid = getpid();
    snprintf(dir, sizeof(dir), "%s%d/", pdir, pid);
  } else {
    char *end;
    int pid = strtol(name, &end, 0);
    if (*name != '\0' 
	&& *end == '\0' 
	&& pid > 0 
	&& (kill(pid, 0) == 0
	    || errno != ESRCH)) {
      pandora_notice("process with pid #" << pid << " still exists");
      pandora_notice("kill it, or choose another name");
      return false;
    }      
    snprintf(dir, sizeof(dir), "%s%s/", pdir, name);
  }
  bool status = create_dir(dir);

  status &= stacks.init(dir, "stacks");
  status &= libconfig.init(dir, "libconfig");
  
  return status;
}

bool Storage::save(storage_type_t type, const text &key, void *data)
{
  if (*dir == '\0') return true;
  if (data == NULL) return false;
  switch(type) {
  case state:   return true;
  case stack:   return save_stack (key, static_cast<StackEntry *>(data));
  case library: return save_lib   (key, static_cast<DynLibrary *>(data));
  case symbol:  return save_symbol(key, static_cast<DynSymbol  *>(data));
  default:      break;
  }

  return false;
}


bool Storage::save_stack(const text &key, StackEntry *se)
{
  bzero(buf, sizeof(buf));
  ostrstream s(buf, sizeof(buf));
  se->print(&s);
  return stacks.write(buf, strlen(buf));
}

bool Storage::save_lib(const text &key, DynLibrary *lib)
{
  bzero(buf, sizeof(buf));
  ostrstream s(buf, sizeof(buf));
  lib->print(&s);
  return libconfig.write(buf, strlen(buf));
}

bool Storage::save_symbol(const text &key, DynSymbol *sym)
{
  bzero(buf, sizeof(buf));
  ostrstream s(buf, sizeof(buf));
  sym->print(&s);
  return libconfig.write(buf, strlen(buf));
}
