/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef THREAD_H
#define THREAD_H

#include <libpandora/global.h>

#ifndef _NOTHREADS
#include <pthread.h>
#endif

#include <libpandora/conf/string.h>
#include <libpandora/conf/time.h>
#include <libpandora/timeval.h>
#include <assert.h>
#include <errno.h>
#include <libpandora/error.h>


#define pthread_error(reason) pandora_error(strerror(reason))

#ifdef _NOTHREADS
#define PTHREAD(FN,ARGS)
#define pthread_t int
#define pthread_mutex_t int
#define pthread_attr_t int
#define pthread_mutexattr_t int
#define pthread_cond_t int
#else
#define PTHREAD(FN,ARGS) { \
  int e= pthread_##FN ARGS; \
  if (e) pthread_error(e); \
}
#endif

extern "C" void *thread_run(void *);

/** Thread encapsulation class.
 */
class Thread {
private:
  pthread_t thread;

public:
  Thread(void) : thread(0) 	      { }
  virtual ~Thread(void) 	      { }

  /// Abstract method run in the created thread.
  virtual void main(void)= 0;

  /// Method to call to effectively execute the thread.
  Thread &run(void);
  Thread &detach(void);
  Thread &join(void);
  Thread &cancel(void);
  inline bool isRunning(void) { return (thread != 0); }
};


#define Mutex PandoraMutex

class Mutex {
protected:
  pthread_mutex_t mx;

public:
#ifdef MUTEX_TYPES
  enum {
#ifdef PTHREAD_MUTEX_FAST_NP
    Fast= PTHREAD_MUTEX_FAST_NP,
#else
    Fast= PTHREAD_MUTEX_NORMAL_NP,
#endif
    Recursive= PTHREAD_MUTEX_RECURSIVE_NP,
    Errorcheck= PTHREAD_MUTEX_ERROR_CHECK_NP
  };
#endif

public:
#ifdef MUTEX_TYPES
#ifndef DEFAULT_MX
  Mutex(void) { 
    PTHREAD(mutex_init,(&mx, 0));
  }
  Mutex(int kind)
#else
  Mutex(int kind= DEFAULT_MX)
#endif
  {
    pthread_mutexattr_t attr;
    PTHREAD(mutexattr_init,(&attr));
#ifdef PTHREAD_MUTEX_FAST_NP
    PTHREAD(mutexattr_setkind_np,(&attr, kind));
#else
    PTHREAD(mutexattr_settype_np,(&attr, kind));
#endif
    PTHREAD(mutex_init,(&mx, &attr));
    PTHREAD(mutexattr_destroy,(&attr));
  }
#else
  Mutex(void) {
    PTHREAD(mutex_init,(&mx, 0));
  }
  Mutex(int /* kind */){
    cerr << "warning: Mutex::Mutex: option not supported, using default" 
	 << endl;
    PTHREAD(mutex_init,(&mx, 0));
  }
#endif

  virtual ~Mutex(void) {
    PTHREAD(mutex_destroy,(&mx));
  }

public:
  Mutex &lock(void) {
    PTHREAD(mutex_lock,(&mx));
    return *this;
  }
  bool tryLock(void) {
#ifndef _NOTHREADS
    int e= pthread_mutex_trylock(&mx);
    switch (e) {
    case 0:	return true;
    case EBUSY:	return false;
    default:	pthread_error(e);
    }
    return false;
#else
    return true;
#endif

  }
  Mutex &unlock(void) {
    PTHREAD(mutex_unlock,(&mx));
    return *this;
  }
};


class Semaphore : public Mutex
{
private:
  int excessSignals;
  pthread_cond_t cv;

public:  
  Semaphore(int xs= 0) : excessSignals(xs) {
    PTHREAD(cond_init,(&cv, 0));
  }
  virtual ~Semaphore(void) {
    PTHREAD(cond_destroy,(&cv));
  }

public:
  void _signal(void) {
    if (++excessSignals < 1) {
      PTHREAD(cond_signal,(&cv));
    }
  }

  void signal(void) {
    lock();
    _signal();
    unlock();
  }

  void _wait(void) {
    if (--excessSignals < 0) {
      PTHREAD(cond_wait,(&cv,&mx));
    }
  }

  void wait(void) {
    lock();
    _wait();
    unlock();
  }

  void _timed_wait(int timeout) {
    if (--excessSignals < 0) {
      timeval tv;
      (void) gettimeofday(&tv, NULL);
      tv = tv + timeout;
      
      timespec ts;
      ts.tv_sec = tv.tv_sec;
      ts.tv_nsec = 1000 * tv.tv_usec;
      
      int e = pthread_cond_timedwait(&cv, &mx, &ts);
      if (e == ETIMEDOUT) {
	pandora_debug("semaphore timed out (timeout = " << timeout << ")");
	e = 0;
      }
      if (e) pthread_error(e);
    }
  }

  void timed_wait(int timeout) {
    lock();
    _timed_wait(timeout);
    unlock();
  }
};


class Condition : public Mutex
{
private:
  pthread_cond_t cv;

public:  
  Condition(void) : Mutex() {
    PTHREAD(cond_init,(&cv, 0));
  }
  virtual ~Condition(void) {
    PTHREAD(cond_destroy,(&cv));
  }

public:
  void signal(void) {
    PTHREAD(cond_signal,(&cv));
  }

  void broadcast(void) {
    PTHREAD(cond_broadcast,(&cv));
  }

  void wait(void) {
    PTHREAD(cond_wait,(&cv,&mx));
  }

  void timed_wait(int timeout) {
    timeval tv;
    (void) gettimeofday(&tv, NULL);
    tv = tv + timeout;
      
    timespec ts;
    ts.tv_sec = tv.tv_sec;
    ts.tv_nsec = 1000 * tv.tv_usec;
    
    int e = pthread_cond_timedwait(&cv, &mx, &ts);
    if (e == ETIMEDOUT) {
      pandora_debug("condition timed out (timeout = " << timeout << ")");
      e = 0;
    }
    if (e) pthread_error(e);
  }
};

#endif /* THREAD_H */


