/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#ifndef URL_H
#define URL_H

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
}

#include <libpandora/text.h>
#include <libpandora/serialize.h>

class URL {
private:
  text 		host;
  text 		absolute;
  u_short 	port;
  
public:
  inline URL(void) : port(0) 			{ }
  inline URL(const char *url) 			{ scan(url); }
  inline URL(const URL &x);
  inline URL &operator =(const URL &x);
  inline ~URL(void) { }

  inline operator long() const;
  inline bool operator ==(const URL &x) const;
  inline bool operator <(const URL &x) const;
  bool isNull(void) 				{ return absolute.isNull(); }

  void escape(void);
  void unescape(void);
  bool isAbsolute(void) 			{ return (!host.isNull()); }
  void scan(const char *url);
  inline bool isOK(void);
  void getURL(bool, char *, int);

  bool makeRequest(char *buf, int len,
		   const char *method, bool useProxy, 
		   time_t ims = 0, const char *hdr = NULL);

  int fetchString(char **str, 
		  in_addr_t proxy_addr = 0, u_short proxy_port = 0);
  int fetchFile(const char *dir, char *name, int len,
		in_addr_t proxy_addr = 0, u_short proxy_port = 0);

  bool ifModifiedSince(time_t, 
		       in_addr_t proxy_addr = 0, u_short proxy_port = 0);
  FILE *get(in_addr_t proxy_addr = 0, u_short proxy_port = 0);

  int sendRequest(const char *method, 
		  in_addr_t proxy_addr = 0, u_short proxy_port = 0,
		  time_t ims = 0, 
		  const char *hdr = NULL);

  in_addr_t getHostAddr(void);
  u_short   getHostPort(void) { return port; }

private:
  static char ascii2char(char c);
  static int fetchHeader(FILE *f);
};

URL::URL(const URL &x) 
  : host(x.host), absolute(x.absolute), port(x.port)
{ 
}

URL &URL::operator =(const URL &x)
{
  host = x.host; absolute = x.absolute; port = x.port;
  return *this;
}

URL::operator long() const
{
  return ((long) host ^ (long) absolute) + port;
}

bool URL::operator ==(const URL &x) const
{
  return (host == x.host) && (absolute == x.absolute) && (port == x.port);
}

bool URL::operator <(const URL &x) const
{
  return 
    (host < x.host) 
    || ((host == x.host) && (absolute < x.absolute))
    || ((host == x.host) && (absolute == x.absolute) && (port < x.port));
}

bool URL::isOK(void) 
{ 
  return !(host.isNull() 
	   || absolute.isNull() 
	   || port == 0); 
}

#endif /* URL_H */
