/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <ctype.h>
}

#include <libpandora/wrapper.h>

#include <libpandora/dynloader.h>
#include <libpandora/pandora.h>
#include <libpandora/stackfactory.h>
#include <libpandora/stackentry.h>
#include <libpandora/compentry.h>
#include <libpandora/fileutil.h>
#include <libpandora/util.h>

extern "C" {
void new_dynloader(const char *lfile) 
{
  char *lib_defs  = read_in_string(lfile);
  new DynLoader();
  if (dynloader != NULL) dynloader->init(lib_defs);
  __FREE(lib_defs); 
}

void delete_dynloader(void) 
{
  if (dynloader != NULL) 
    delete dynloader;
}

void new_pandora(const char *sfile) 
{
  char *stack_defs = read_in_string(sfile);
  new Pandora();
  if (pandora != NULL) pandora->init(stack_defs);
  __FREE(stack_defs);
}

void delete_pandora(void) 
{
  if (pandora != NULL) delete pandora;
}

stack_handle_t pandora_start(const char *stk, int threaded)
{
  if (pandora == NULL) return NIL_STACK_HANDLE;
  return pandora->start(text(stk), threaded, true);
}

int pandora_stop(stack_handle_t sh)
{
  if (pandora == NULL) return -1;
  return pandora->stop(sh);
}

void pandora_poll(void)
{
  if (pandora != NULL) pandora->poll();
}

void pandora_poll_stack(stack_handle_t sh)
{
  if (pandora != NULL) pandora->poll(sh);
}

int pandora_suspend(stack_handle_t sh)
{
  if (pandora == NULL) return -1;
  return pandora->stop(sh);
}

int pandora_resume(stack_handle_t sh)
{
  if (pandora == NULL) return -1;
  return pandora->stop(sh);
}

stack_handle_t pandora_use_stack(const char *stk)
{
  if (pandora == NULL) return NIL_STACK_HANDLE;
  return pandora->use(stk);
}

int pandora_set_stack(const char *str)
{
  if (pandora == NULL) return -1;
  StackFactory sf(str);
  StackEntry *se = sf.parseStack();
  int ret = pandora->setStack(*se, true);
  __DELETE(se);
  return ret;
}

int pandora_set_option(const char *stk, const char *comp, const char *op,
		       const char *val)
{
  if (pandora == NULL) return -1;

  MultiValue mv;
  if (val != NULL) {
    StackFactory sf(val);
    if (sf.parseValue(&mv) <= 0) {
      pandora_warning("failed parsing value: " << val);
      return -1;
    }
  }
  return (int)pandora->setOption(text(stk), text(comp), text(op), mv);
}

int pandora_set_live_option(stack_handle_t sh, const char *comp, 
			    const char *op, const char *val)
{
  if (pandora == NULL) return -1;

  MultiValue mv;
  if (val != NULL) {
    StackFactory sf(val);
    if (sf.parseValue(&mv) <= 0) {
      pandora_warning("failed parsing value: " << val);
      return -1;
    }
  }
  return (int)pandora->setOption(sh, text(comp), text(op), mv);
}

int dynloader_register_symbol(const char *name, const char *lib, 
			      const char *prefix) 
{
  if (dynloader == NULL) return -1;
  return dynloader->registerSymbol(text(name), text(lib), text(prefix));
}

int dynloader_register_library(const char *id, long version, const char *loc,
			       const char **d, int n) 
{
  if (dynloader == NULL) return -1;
  text *deps = new text[n];
  for (int i = 0; i < n; ++i) deps[i].init(d[i]);
  bool ret =  (dynloader->registerLibrary(id, version, loc) 
	       && dynloader->registerDependencies(id, deps, n));
  __DELETE_ARRAY(deps);
  return ret;
}

}
