/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include "cacheevalcomponent.h"
#include <pandora_components/cachetranspacket.h>
#include <pandora_components/icptranspacket.h>
#include <pandora_components/valuepacket.h>

component_export(CacheEvalComponent, CacheEvalPacket, FloatValuePacket);

CacheEvalComponent::CacheEvalComponent(void) 
  : global(0), temp(0),
    coeff_rtt(0.0), coeff_lat(0.0), 
    coeff_bytes(0.0), coeff_coop(0.0), coeff_stale(0.0), 
    mem(1), avg(1), firstTime(0), target(0),
    abstime(true), offset(0), result(0.0)
{
  registerOption("rtt", &coeff_rtt);
  registerOption("lat", &coeff_lat);
  registerOption("bytes", &coeff_bytes);
  registerOption("coop", &coeff_coop);
  registerOption("stale", &coeff_stale);
  registerOption("avg", &avg);
  registerOption("mem", &mem);
  registerOption("abs", &abstime);
  registerOption("offset", &offset);
}

bool CacheEvalComponent::add(Packet *pkt) 
{
  CacheEvalPacket *c = static_cast<CacheEvalPacket *>(pkt);
  //if (c->docs == 0) return_clean(pkt);

  time_t lTime = c->timeStamp.tv_sec;
  temp.add(*c);
  cleanPacket(c);

  if (firstTime == 0) {
    firstTime = lTime;
    target = firstTime + avg;
  }
  
  while (lTime >= target) {
    //pandora_debug((target-firstTime)/avg);
    result += eval(temp);
    Q.add(temp);
    temp.reset();

    if ((target - firstTime) > (mem *avg)) {
      CacheEvalPacket rem = Q.remove();
      result -= eval(rem);
    }

    update();

    target += avg;
  }

  return false;
}

double CacheEvalComponent::eval(CacheEvalPacket &c)
{
  if (c.docs < 1) return 0.0;

#if 1
  pandora_assert ((coeff_rtt + coeff_lat 
		   + coeff_bytes /*+ coeff_stale */) != 0.0);

  double rtt = ((double)c.rtt_saved) / ((double)c.rtt);
  double lat = ((double)c.lat_saved) / ((double)c.lat);
  double bytes = ((double)c.bytes_saved) / ((double)c.bytes);
  //double stale = ((double)c.stale) / ((double)c.docs);

  double res = (((coeff_rtt * rtt) 
		 + (coeff_lat * lat)
		 + (coeff_bytes * bytes)
		 /*+ (coeff_stale * stale)*/)
		/ (coeff_rtt + coeff_bytes + coeff_lat /*+ coeff_stale*/));
#else
  //pandora_info(c.bytes_coop);
  double res = (double)c.bytes_saved/(double)(c.bytes);
  //double res = (double)c.bytes_saved / (double)c.bytes_coop;
  //double res = (c.docs_hit > 0) ? (double)c.stale / (double)c.docs_hit : 0.0;
#endif

  return res;
}


void CacheEvalComponent::update(void) 
{      
  int n = Q.size();
  double res = result/n;

  timeval ts;
  ts.tv_sec = target - (n*avg/2);
  ts.tv_usec = 0;
  if (!abstime) ts.tv_sec -= firstTime;
  ts.tv_sec += offset;
  
  FloatValuePacket *vp = new FloatValuePacket(res, ts);
  push(vp);
}
