/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <pandora_components/cacheevalpacket.h>
#include <pandora_components/cachetranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/httptranspacket.h>
#include <pandora_components/icptranspacket.h>
#include <libpandora/serialize.h>
#include <libpandora/timeval.h>

packet_export(CacheEvalPacket, ICPTransPacket|CacheTransPacket);

#define VALFILTER 0

CacheEvalPacket::CacheEvalPacket(int) 
  : rtt(0), rtt_saved(0), lat(0), lat_saved(0), 
    bytes(0), bytes_saved(0), bytes_coop(0),
    stale(0), 
    docs(0), docs_hit(0), docs_miss(0), docs_sibling(0)
{
  timeStamp.tv_sec = 0;
  timeStamp.tv_usec = 0;    
}

CacheEvalPacket::CacheEvalPacket(CacheTransPacket *ctpp) 
  : rtt(0), rtt_saved(0), 
    lat(0), lat_saved(0), 
    bytes(0), bytes_saved(0), bytes_coop(0),
    stale(0), 
    docs(0), docs_hit(0), docs_miss(0), docs_sibling(0)
{
  int cbytes = 0;
  timeStamp = ctpp->timeStamp;

  locatePacket(HTTPTransPacket,	httptp,	ctpp);
  locatePacket(HTTPPacket,	httpp,	httptp);
  //if ((httpp->code / 100) > 3) goto finished;

  cbytes = (httptp->reqhlen + httptp->reqmlen
	    + httptp->resphlen + httptp->respmlen);

#if VALFILTER
#if 0
  if (ctpp->type == CacheTransPacket::miss && ctpp->bytes_saved < -100) {
    pandora_info("correcting bytes:saved: " 
		 << ctpp->bytes_saved << " -> "
		 << -(cbytes+ctpp->bytes_saved));
    ctpp->bytes_saved = -(cbytes+ctpp->bytes_saved);
  }
#endif

  if ((int) httptp->rtt <= 0) {
    //pandora_warning("negative rtt: " << httptp->rtt);
    goto finished;
  }
  if ((int) httptp->lat <= 0) {
    //pandora_warning("negative lat: " << httptp->lat);
    goto finished;
  }
  if ((int) httptp->rtt < (int) httptp->lat) {
    //pandora_warning("smaller rtt: " << httptp->rtt << " / " << httptp->lat);
    goto finished;
  }
  if ((int) ctpp->rtt_saved < (int) ctpp->lat_saved) {
    //pandora_warning("smaller rtt_saved: " << ctpp->rtt_saved
    //<< " / " << ctpp->lat_saved);
    goto finished;
  }
  if ((int) cbytes < 100) {
    //pandora_warning("negative bytes: " << cbytes);
    goto finished;
  }
  if ((int) ctpp->rtt_saved > 10000 
      || (int) ctpp->rtt_saved < -10000  ){
    //pandora_warning("too much rtt saved: " << ctpp->rtt_saved);
    goto finished;
  }
  if ((int) ctpp->lat_saved > 10000 
      || (int) ctpp->lat_saved < -10000  ){
    //pandora_warning("too much lat saved: " << ctpp->lat_saved);
    goto finished;
  }
  if ((int) ctpp->bytes_saved > cbytes
      || (int) ctpp->bytes_saved  < -100  ){
    //pandora_warning("too few bytes saved: " 
    //		    << cbytes << " / " << ctpp->bytes_saved );
    goto finished;
  }
  if ((int) cbytes < (int) ctpp->bytes_saved) {
    //pandora_warning("too few bytes saved: " 
    //		    << cbytes << " < " << ctpp->bytes_saved );
    goto finished;
  }
#endif
  
  switch (ctpp->type) {
  case CacheTransPacket::miss:
#if VALFILTER
    if ((int) httptp->rtt < (int) ctpp->rtt_saved)
      goto finished;
    if ((int) ctpp->bytes_saved > 0) 
      goto finished;
#endif

    docs_miss = 1;
    rtt = httptp->rtt; 
    rtt_saved = ctpp->rtt_saved;
    lat = httptp->lat;
    lat_saved = ctpp->lat_saved;
    bytes = cbytes;
    bytes_saved = ctpp->bytes_saved;
    break;
    
  case CacheTransPacket::hit_sibling:
    ++docs_sibling;
    bytes_coop = ctpp->bytes_sibling;
    // fall through
    
  case CacheTransPacket::hit_ok:   case CacheTransPacket::hit_bad:
#if 0
    // XXX fix for HEAD and IMS requests
        if ((int) ctpp->bytes_saved 
	< (int)(cbytes - httptp->reqhlen - httptp->respmlen)) {
	  pandora_debug("saved: " << ctpp->bytes_saved 
			<< ", cbytes: " << cbytes);
	  ctpp->bytes_saved = cbytes;
    }
#endif
    ++docs_hit;
    rtt = httptp->rtt; rtt_saved = ctpp->rtt_saved;
    lat = httptp->lat; lat_saved = ctpp->lat_saved;
    bytes = cbytes;
    bytes_saved = ctpp->bytes_saved;

    if ((ctpp->p_lm > 0) && (httpp->lm > 0) 
	&& (ctpp->p_lm > httpp->lm)
	&& (ctpp->p_lm < ctpp->timeStamp.tv_sec))
      ++stale;
    //pandora_debug(stale << " / " << docs_hit);
    break;
    
  case CacheTransPacket::miss_sibling:		
  case CacheTransPacket::alone_sibling:		
  case CacheTransPacket::avoided:		
  case CacheTransPacket::alone: 
  default:					goto finished;
  }  
  ++docs;

 finished:
#if VALFILTER
  //if (docs == 0) ctpp->log();
#endif
  packetSetUp(ctpp);
}


CacheEvalPacket::CacheEvalPacket(const CacheEvalPacket &x) 
  : Packet(x), rtt(x.rtt), rtt_saved(x.rtt_saved),
    lat(x.lat), lat_saved(x.lat_saved),
    bytes(x.bytes), bytes_saved(x.bytes_saved), bytes_coop(x.bytes_coop),
    stale(x.stale), 
    docs(x.docs), docs_hit(x.docs_hit), docs_miss(x.docs_miss), 
    docs_sibling(x.docs_sibling) 
{
}

CacheEvalPacket &CacheEvalPacket::operator=(const CacheEvalPacket &x) 
{
  Packet::operator=(x);
  rtt = x.rtt; rtt_saved = x.rtt_saved;
  lat = x.lat; lat_saved = x.lat_saved;
  bytes = x.bytes; bytes_saved = x.bytes_saved; bytes_coop = x.bytes_coop;
  stale = x.stale; 
  docs = x.docs; docs_hit = x.docs_hit; docs_miss = x.docs_miss; 
  docs_sibling = x.docs_sibling;
  return *this;
}

CacheEvalPacket::CacheEvalPacket(ICPTransPacket *itpp) :
  rtt(0), rtt_saved(0), 
  lat(0), lat_saved(0), 
  bytes(0), bytes_saved(0), bytes_coop(0),
  stale(0), 
  docs(0), docs_hit(0), docs_miss(0), docs_sibling(0)
{
  timeStamp = itpp->timeStamp;
  pandora_assert(itpp->length > 0);
  bytes_coop = itpp->length;

  packetSetUp(itpp);
}

void CacheEvalPacket::reset(void) 
{
  rtt = 0; rtt_saved = 0; 
  lat = 0; lat_saved = 0; 
  bytes = 0; bytes_saved = 0; bytes_coop = 0;
  stale = 0; 
  docs = 0; docs_hit = 0; docs_miss = 0; docs_sibling = 0;
}

void CacheEvalPacket::add(const CacheEvalPacket &x) 
{
  if (compareTimeStamp(x.timeStamp, timeStamp) > 0) 
    timeStamp = x.timeStamp;
  rtt += x.rtt; rtt_saved += x.rtt_saved;
  lat += x.lat; lat_saved += x.lat_saved;
  bytes += x.bytes; bytes_saved += x.bytes_saved; bytes_coop += x.bytes_coop;
  stale += x.stale; 
  docs += x.docs; docs_hit += x.docs_hit; docs_miss += x.docs_miss; 
  docs_sibling += x.docs_sibling;
}

void CacheEvalPacket::sub(const CacheEvalPacket &x) 
{
  rtt -= x.rtt; rtt_saved -= x.rtt_saved;
  lat -= x.lat; lat_saved -= x.lat_saved;
  bytes -= x.bytes; bytes_saved -= x.bytes_saved; bytes_coop -= x.bytes_coop;
  stale -= x.stale; 
  docs -= x.docs; docs_hit -= x.docs_hit; docs_miss -= x.docs_miss; 
  docs_sibling -= x.docs_sibling;
}

void CacheEvalPacket::print(ostream *f) 
{
  *f << timeStamp << '\t' << "CACHE EVAL " 
    //<< rtt_saved << '/' << rtt << ' '
    //<< lat_saved << '/' << lat << ' '
     << bytes_saved << "\t/\t" << bytes //<< " (" << bytes_coop << ") "
    //<< stale << ' '
     << docs 
     << " (" << docs_hit << "," << docs_miss << "," << docs_sibling << ") "
     << endl;
}

size_t CacheEvalPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(rtt);
  serialVar(rtt_saved);
  serialVar(lat);
  serialVar(lat_saved);
  serialVar(bytes);
  serialVar(bytes_saved);
  serialVar(bytes_coop);
  serialVar(stale);
  serialVar(docs);
  serialVar(docs_hit);
  serialVar(docs_miss);
  serialVar(docs_sibling);

  return count;
}

size_t CacheEvalPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(rtt);
  unserialVar(rtt_saved);
  unserialVar(lat);
  unserialVar(lat_saved);
  unserialVar(bytes);
  unserialVar(bytes_saved);
  unserialVar(bytes_coop);
  unserialVar(stale);
  unserialVar(docs);
  unserialVar(docs_hit);
  unserialVar(docs_miss);
  unserialVar(docs_sibling);

  return count;
} 
