/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "cachematchcomponent.h"
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/timeval.h>

component_export(CacheMatchComponent, HTTPTransPacket, HTTPPacket|CacheTransPacket);

CacheMatchComponent::CacheMatchComponent(void) 
  : cache(NULL), cto(0), check(false)
{
  registerOption("cache", &cto);
  registerOption("check", &check);
}

bool CacheMatchComponent::add(Packet* obj)
{
  HTTPTransPacket *msg = static_cast<HTTPTransPacket *>(obj);

  HTTPTransPacket::httpt_pos_t pos = msg->pos;

  if (pos == HTTPTransPacket::alone) {
    push(new CacheTransPacket(msg));
    return false;
  }

  int rindex = (int) (pos == HTTPTransPacket::after);
  req_list_t *list0=&rlist[rindex];
  req_list_t *list1=&rlist[1 - rindex];
  
  locatePacket(HTTPPacket,	httpp,	msg);

  if (httpp->coll > 0) {
    //pandora_debug("[reemitted request]");
    if (cache == NULL) {
      setCacheRequest(msg);
      flush(list0);
    } else {
      cleanPacket(msg);
    }
    return false;
  }

  if ((cache == NULL) && (pos == HTTPTransPacket::after)) {
    setCacheRequest(msg);
  }      

  bool found = false;
  
  if (!list1->empty()) {
    switch (pos) {
    case HTTPTransPacket::before: 
    case HTTPTransPacket::sibling:
      found = findMatch(msg, list1, true);
      break;	
    case HTTPTransPacket::after: 
      found = findMatch(msg, list1, false);
      break;
    case HTTPTransPacket::alone: 
    case HTTPTransPacket::matched:
    default:
      pandora_warning("incorrect message");
      break;
    }
  }

  if (!found) list0->push_back(msg);
  
  return false /* (list0->empty() && list1->empty()) */;
}


bool CacheMatchComponent::prepare(void)
{
  if (!check 
      || cache != NULL 
      || rlist[0].empty()) 
    return true;

  locatePacket(HTTPPacket, cachebuf, (rlist[0]).front());
  if (cachebuf == NULL) return true;

  HTTPPacket *tmp = new HTTPPacket(*cachebuf);
  locatePacket(IPPacket, ipp, tmp);
  if (ipp == NULL) return true;
  (ipp->dst).s_addr = 0;

  push(tmp, 1);
  if (cto > 0) timeout = cto;
  check = false;

  return false;
}

void CacheMatchComponent::cleanup(void)
{
  locatePacket(HTTPPacket, httpp, cache);
  if (httpp != NULL) httpp->coll = 1;

  flush(&rlist[0]);
  flush(&rlist[1]);
  flushCacheRequest();
}

void CacheMatchComponent::flush(req_list_t *l)
{
  while (l->size() > 0) {
    HTTPTransPacket *httptp = l->front();
    pandora_assert(httptp != NULL);
    l->pop_front();
    push(((httptp->pos != HTTPTransPacket::after) && (cache != NULL))
      ? new CacheTransPacket(httptp, cache)
      : new CacheTransPacket(httptp));
  }
  //pandora_assert(l.empty())
} 

void CacheMatchComponent::setCacheRequest(HTTPTransPacket *httptp)
{
  if (cache == NULL) {
    cache = httptp;
    cache->refer();
#if 0
    locatePacket(HTTPPacket, httpp, cache);
    pandora_debug(lastTime
		  << " [caching request for:  " 
		  << httpp->url << "]");
#endif
  }
}

void CacheMatchComponent::flushCacheRequest(void)
{
  if (cache != NULL) {
#if 0
    locatePacket(HTTPPacket, httpp, cache);
    pandora_debug(lastTime
		  << " [flushing request for: " 
		  << httpp->url << "]");
#endif
    cache->release();
    cleanPacket(cache);
  }
}

bool CacheMatchComponent::findMatch(HTTPTransPacket *httptp, 
				    req_list_t *l, bool b)
{
  req_list_t::iterator ptr;
  HTTPTransPacket *before = NULL, *after = NULL;

  if (b) before = httptp;
  else after = httptp;
  
  for (ptr = l->begin(); ptr != l->end(); ++ptr) {
    HTTPTransPacket *httptp1 = *ptr;

    if (b) after = httptp1;
    else before = httptp1;

    locatePacket(IPPacket, bip, before);
    locatePacket(IPPacket, aip, after);

    if ((before->timeStamp < after->timeStamp)
	& ((bip->dst).s_addr == (aip->src).s_addr)) {
      push(new CacheTransPacket(before, after));
      l->erase(ptr);
      return true;
    }
  }
  return false;
}
