/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/cachetranspacket.h>
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/util.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>
#include <libpandora/timeval.h>

packet_export(CacheTransPacket, HTTPTransPacket);

CacheTransPacket::CacheTransPacket(HTTPTransPacket* before, 
				   HTTPTransPacket* after) 
  : type(undefined), rtt_saved(0), lat_saved(0),
    bytes_saved(0), bytes_sibling(0),
    p_ims((time_t) -1), p_ius((time_t) -1),
    p_lm((time_t) -1), p_expires((time_t) -1)
{ 
  locatePacket(HTTPPacket,  	hbefore,  	before);
  locatePacket(TCPPacket,  	tbefore,  	hbefore);
  locatePacket(IPPacket,  	ibefore,  	tbefore);
  locatePacket(HTTPPacket,  	hafter,  	after);
  locatePacket(TCPPacket,  	tafter,  	hafter);
  locatePacket(IPPacket,  	iafter,  	tafter);

  if (iafter == NULL || ibefore == NULL) return;

  type = ((hafter->coll == 0) ? miss : hit_ok);
  
  proxy.s_addr = ibefore->dst.s_addr;
  p_dport = tbefore->dport;
  p_sport = tafter->sport;
  p_ims = hafter->ims;
  p_ius = hafter->ius;
  p_lm = hbefore->lm;
  p_expires = hbefore->expires;

  rtt_saved = after->rtt - before->rtt;
  lat_saved =  after->lat - before->lat;
   
  ibefore->dst.s_addr = iafter->dst.s_addr;
  tbefore->dport = tafter->dport;
  if (type == hit_ok) {
    hbefore->lm = hafter->lm;
    hbefore->expires = hafter->expires;
    before->rtt = after->rtt;
    before->lat = after->lat;
#if 0
    before->resphlen = after->resphlen;
    before->respmlen = after->respmlen;
    before->reqhlen = after->reqhlen;
    before->reqmlen = after->reqmlen;
#endif
  } 
    
  if (type == miss) {
    bytes_saved = (before->reqhlen + before->reqmlen
		   + before->resphlen+before->respmlen) 
      - (after->reqhlen + after->reqmlen 
	 + after->resphlen+after->respmlen);
  } else { 
    bytes_saved = after->reqhlen + after->reqmlen
      + after->resphlen + after->respmlen;
  }

  before->pos = HTTPTransPacket::matched;
  timeStamp = before->timeStamp;
  packetSetUp(before);
  cleanPacket(after);
}

CacheTransPacket::CacheTransPacket(HTTPTransPacket* httptp) :
  type(undefined), rtt_saved(0), lat_saved(0),
  bytes_saved(0), bytes_sibling(0),
  p_ims((time_t) -1), p_ius((time_t) -1),
  p_lm((time_t) -1), p_expires((time_t) -1)
{
  if (httptp == NULL) return;

  timeStamp = httptp->timeStamp;
  packetSetUp(httptp);

  switch (httptp->pos) {
  case HTTPTransPacket::before:		type = hit_bad; 	break;
  case HTTPTransPacket::sibling:	type = alone_sibling; 	break;
  case HTTPTransPacket::after:		type = avoided; 	return;
  case HTTPTransPacket::alone:		type = alone; 		return;
  case HTTPTransPacket::matched:	break;
  default:
    pandora_warning("incorrect message: #" << (int) httptp->pos);
    return;
  }

  locatePacket(HTTPPacket, 	httpp, 	httptp);
  locatePacket(TCPPacket, 	tcpp, 	httpp);
  locatePacket(IPPacket, 	ipp, 	tcpp);
  
  if (type == hit_bad) {
    proxy.s_addr = ipp->dst.s_addr;
    p_dport = tcpp->dport;
    
    lat_saved = httptp->lat / 2; 
    rtt_saved = httptp->rtt - lat_saved;
    bytes_saved = httptp->reqhlen + httptp->reqmlen 
      + httptp->resphlen + httptp->respmlen;
  } else {
    proxy.s_addr = ipp->src.s_addr;
    p_sport = tcpp->sport;
  }    
}

CacheTransPacket::CacheTransPacket(const CacheTransPacket& x) : 
  Packet(x), type(x.type), proxy(x.proxy), 
  p_sport(x.p_sport), p_dport(x.p_dport),
  rtt_saved(x.rtt_saved), lat_saved(x.lat_saved), 
  bytes_saved(x.bytes_saved), bytes_sibling(bytes_sibling),
  p_ims(x.p_ims), p_ius(x.p_ius),
  p_lm(x.p_lm), p_expires(x.p_expires) 
{
}

CacheTransPacket& CacheTransPacket::operator= (const CacheTransPacket& x) 
{
  Packet::operator=(x);
  type = x.type; proxy = x.proxy;
  p_sport = x.p_sport; p_dport = x.p_dport;
  rtt_saved = x.rtt_saved; lat_saved = x.lat_saved;
  bytes_saved = x.bytes_saved; bytes_sibling = x.bytes_sibling;
  p_ims = x.p_ims; p_ius = x.p_ius;
  p_lm = x.p_lm; p_expires = x.p_expires;       
  
  return *this;
}

void CacheTransPacket::print(ostream *f)
{
  locatePacket(HTTPTransPacket, httptp, this);
  locatePacket(HTTPPacket, 	httpp, 	httptp);
  locatePacket(TCPPacket, 	tcpp, 	httpp);
  locatePacket(IPPacket, 	ipp, 	tcpp);
  if (ipp == NULL) {
    pandora_error("cannot find underlying IP packet");
    return;
  }

  if (httptp->reqOK()) {
    *f << timeStamp << ' '
       << setw(6) << setfill(' ') << httptp->rtt << ' ' << httptp->lat << ' '
       << '(' << rtt_saved << ' ' << lat_saved << ") " 
       << ipp->src << ':'  << ntohs(tcpp->sport) << ' '
       << ipp->dst << ':' << ntohs(tcpp->dport) << ' ';
    if (httptp->respOK()) {
      *f << httpp->code << ' '
	 << (httptp->resphlen + httptp->respmlen) << " ";
    }

    switch(httpp->method){
    case HTTPPacket::GET:     *f << "GET " ;     break;
    case HTTPPacket::POST:    *f << "POST " ;    break;
    case HTTPPacket::HEAD:    *f << "HEAD " ;    break;
    case HTTPPacket::PUT:     *f << "PUT " ;     break;
    case HTTPPacket::OPTIONS: *f << "OPTIONS " ; break;
    case HTTPPacket::TRACE:   *f << "TRACE " ;   break;
    case HTTPPacket::DELETE:  *f << "DELETE " ;  break;
    case HTTPPacket::UNDEF: default:             break;
    }
    if (!httpp->hashed) *f << httpp->url;
    else *f << " [hashed] ";

    *f << " HTTP/";
    switch(type) {
    case avoided:	*f << "AVOIDED ";	      		break;
    case miss:		*f << "MISS ";    			break;
    case miss_sibling:	*f << "MISS(sibling) ";    		break;
    case hit_ok:	*f << "HIT(ok) ";  			break;
    case hit_bad:	*f << "HIT(bad) ";   			break;
    case hit_sibling:	*f << "HIT(sibling) ";			break;
    case alone:		*f << "ALONE ";				break;
    case alone_sibling:	*f << "ALONE(sibling) ";      		break;
    default:		*f << "?? " << (int)type << ' ';   	break;
    }

    *f << (httpp->absolute ? 'A' : '-' )
       << (httpp->dynamic ? 'D' : '-' )
       << (httpp->gap ? 'G' : '-' ) 
       << (httpp->pragma == HTTPPacket::no_cache ? "N" : "-")
       << (httpp->coll > 0 ? 'P' : '-');    

    *f << endl;
  }
}

size_t CacheTransPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(proxy);
  serialVar(p_sport);
  serialVar(p_dport);
  serialVar(rtt_saved);
  serialVar(lat_saved);
  serialVar(bytes_saved);
  serialVar(bytes_sibling);
  serialVar(p_ims);
  serialVar(p_ius);
  serialVar(p_lm);
  serialVar(p_expires);
  
  return count;
}


size_t CacheTransPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(proxy);
  unserialVar(p_sport);
  unserialVar(p_dport);
  unserialVar(rtt_saved);
  unserialVar(lat_saved);
  unserialVar(bytes_saved);
  unserialVar(bytes_sibling);
  unserialVar(p_ims);
  unserialVar(p_ius);
  unserialVar(p_lm);
  unserialVar(p_expires);

  return count;
}

bool CacheTransPacket::sibling_update(HTTPTransPacket *htp)
{
  if (htp == NULL) return false;
  
  switch(type) {
  case miss: 	type = miss_sibling; 		break;
  case hit_ok:  // fall through
  case hit_bad: type = hit_sibling;		break;
  case avoided: 				break;
  default:
    pandora_warning("invalid type: #" << (int)type); 
    break;
  }

  bytes_sibling += (htp->reqhlen + htp->reqmlen
		    + htp->resphlen + htp->respmlen);

  cleanPacket(htp);
  return true;
}

extern_pandora(algo, bool, cachesibip, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(CacheTransPacket, ctpp, pkt);
  locatePacket(IPPacket, ipp, ctpp);
  if (ipp == NULL) return false;

  k->set((ctpp->type == CacheTransPacket::hit_sibling)
	 ? (ipp->dst).s_addr
	 : (ctpp->proxy).s_addr);
 
  return true;
}

extern_pandora(algo, bool, cachesibport, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(CacheTransPacket, ctpp, pkt);
  locatePacket(TCPPacket, 	  tcpp, ctpp);
  locatePacket(IPPacket, 	  ipp,  tcpp);
  if (ipp == NULL) return false;

  if (ctpp->type == CacheTransPacket::hit_sibling) {
    k->set(ctpp->proxy.s_addr, ipp->dst.s_addr, ctpp->p_sport, tcpp->dport);
  } else {
    k->set(ipp->src.s_addr, ctpp->proxy.s_addr, tcpp->sport, ctpp->p_dport);
  }

  return true;
}
