/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/dnstranspacket.h>
#include <pandora_components/dnspacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>

packet_export(DNSTransPacket, DNSPacket);

DNSTransPacket::DNSTransPacket(DNSPacket *c, DNSPacket *r) 
  : type(undefined), rtt(-1), length(-1)
{
  if (c == NULL || r == NULL) {
    cleanPacket(c);
    cleanPacket(r);
    return;
  }
 
  type = match;
  
  c->aa = r->aa;
  c->tc = r->tc;
  c->ra = r->ra;
  c->rcode = r->rcode;
  
  if (r->an.size() > 0) c->an = r->an;
  if (r->ns.size() > 0) c->ns = r->ns;
  if (r->ar.size() > 0) c->ar = r->ar;
  
  rtt = diffTimeStamp(r->timeStamp, c->timeStamp);
  length = c->length + r->length;

  timeStamp = c->timeStamp;
  
  packetSetUp(c);
  cleanPacket(r);
}

DNSTransPacket::DNSTransPacket(DNSPacket *pkt) 
  : type(undefined), rtt(-1), length(-1)
{
  if (pkt == NULL) return;
  
  type = (pkt->isQuery()) ? request : response;
  
  timeStamp = pkt->timeStamp;
  
  packetSetUp(pkt);
}

DNSTransPacket::DNSTransPacket(const DNSTransPacket& x) 
  : Packet(x), type(x.type), rtt(x.rtt), length(x.length) 
{
}

DNSTransPacket& DNSTransPacket::operator= (const DNSTransPacket& x) 
{
  Packet::operator=(x);
  type = x.type; rtt = x.rtt; length = x.length;
  return *this;
}

void DNSTransPacket::print(ostream *f) 
{
  locatePacket(DNSPacket, 	dnsp, 	this);
  locatePacket(IPPacket,	ipp,	dnsp);

  if (type != match) {
    if (dnsp != NULL) dnsp->print(f);
    return;
  }

  *f << timeStamp << ' '
     << setw(6) << setfill(' ') << rtt << ' '
     << ipp->src << ' ' << ipp->dst << ' '
     << length << ' '
     << "[dns] ";
  if (dnsp->qd.size() > 0) *f << dnsp->qd[0].name << ' ';
  else *f << "? ";
    
  if (dnsp->an.size() > 0) {
    for (int i=0; i < dnsp->an.size(); ++i) {
      *f << ">>> " ;
      dnsp->printRR(f, dnsp->an[i]);
      *f << " <<<";
      *f << ' ';
    }
  }
  else *f << "- ";    
    
  if (dnsp->ns.size() > 0) {
    for (int i=0; i < dnsp->ns.size(); ++i) {
      dnsp->printRR(f, dnsp->ns[i]);
      *f << ' ';
    }
  }
  else *f << "- ";    

  if (dnsp->ar.size() > 0) {
    for (int i=0; i < dnsp->ar.size(); ++i) {
      dnsp->printRR(f, dnsp->ar[i]);
      *f << ' ';
    }
  }
  else *f << "- ";
    
  *f << (int)dnsp->opcode << ':' << (int)dnsp->rcode << ' '
     << ((dnsp->aa) ? 'A' : '-')
     << ((dnsp->tc) ? 'T' : '-') << ' ';
  *f << endl;
}

size_t DNSTransPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(rtt);
  serialVar(length);

  return count;
}

size_t DNSTransPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(rtt);
  unserialVar(length);

  return count;
}
