/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <libpandora/global.h>

extern "C" {
#include <sys/stat.h>
#include <libgen.h>
}

#include "faminputcomponent.h" 
#include <pandora_components/fseventpacket.h> 
#include <pandora_components/valuepacket.h> 
#include <libpandora/conf/poll.h>
#include <libpandora/conf/snprintf.h>

component_export(FAMInputComponent, TextValuePacket+, FSEventPacket);

int FAMInputComponent::request_t::count = 0;

FAMInputComponent::FAMInputComponent(void)
  : doStop(false), initialdir(NULL), block(false)
{ 
  registerOption("dir", &initialdir); 
}

int FAMInputComponent::init(void) 
{
    if (FAMOpen(&fc) < 0) {
      pandora_warning("could not open connection to FAM daemon");
      return ERROR_FILENO;
    }

    if ((initialdir != NULL) && (monitor(request_t(initialdir)) < 0)) 
      return ERROR_FILENO;

    return FAMCONNECTION_GETFD(&fc);
}

bool FAMInputComponent::process(void) 
{
  struct pollfd pfd;

  pfd.fd = FAMCONNECTION_GETFD(&fc);
  pfd.revents = 0;
  pfd.events = POLLIN;
  
  int pres = poll(&pfd, 1, 200);
  if (pres < 0) {
    if (errno == EINTR) return false;
    pandora_pwarning("poll");
    return true;
  }

  if ((pres == 0) && toMonitor.isEmpty()) return false;
  
  do {
    while (FAMPending(&fc)) {
      FAMEvent fe; 
      if (FAMNextEvent(&fc, &fe) < 0) {
	pandora_warning("cannot get next FAM event");
	return true;
      }
      handle(&fe);
    }
    
    if (block) break;

    if (!toMonitor.isEmpty()) {
      request_t *req = toMonitor.get();
      if (req == NULL) return true;
      monitor(*req);
      __DELETE(req);
    }
  } while (!toMonitor.isEmpty());

  return false;
}

void FAMInputComponent::finish(void) 
{
  while(!toMonitor.isEmpty()) {
    request_t *req = toMonitor.get();
    __DELETE(req);
  }
  FAMClose(&fc);
  doStop = false;
}

void FAMInputComponent::halt(void) 
{
  doStop = true;
  toMonitor.put(NULL);
}

int FAMInputComponent::monitor(const request_t &req)
{
  if (req.filename.length() <= 0) return -1;

  text dir;

  if (realpath(req.filename.data(), path_buf) == NULL) {
    pandora_pwarning(req.filename);
    return 0;
  }
  dir.init(path_buf);

  int mode;
  if ((mode = getDirName(dir)) < 0)	return -1;
  if (monitored_dirs.includesVal(dir))	return 0;
  
  FAMRequest fr;
  if (FAMMonitorDirectory(&fc, dir.data(), &fr, NULL) < 0) {
    pandora_warning("failed to monitor: " << req.filename);
    return -1;
  }
  
  block = true;

  monitored_dirs.atPut(fr.reqnum+1, dir);

  pandora_debug("[monitored directories: " << monitored_dirs.size() << "]");

  return fr.reqnum+1;
}

bool FAMInputComponent::cancel(int req)
{
  FAMRequest fr;
  fr.reqnum = req-1;

  if (FAMCancelMonitor(&fc, &fr) < 0) {
    pandora_warning("failed cancelling request #" << req);
    return -1;
  }
  monitored_dirs.removeKey(req);

  pandora_debug("[monitored directories: " << monitored_dirs.size() << "]");

  return true;
}

bool FAMInputComponent::handle(FAMEvent *fe)
{
  if (fe == NULL) return false;

  text fname;
  if (!getFileName(fe, fname)) return false;

  FSEventPacket *fsep = NULL;

  switch (fe->code) {
  case FAMChanged:
    fsep = new FSEventPacket(fname.data(), FSEventPacket::MODIFIED);
    break;

  case FAMDeleted: {
    int req = monitored_dirs.keyAtOrNil(fname);
    if (req > 0) cancel(req);
    fsep = new FSEventPacket(fname.data(), FSEventPacket::DELETED);
    } break;

  case FAMCreated:
    toMonitor.put(new request_t(fname));
    fsep = new FSEventPacket(fname.data(), FSEventPacket::CREATED);
    break;

  case FAMExists: 	
    toMonitor.put(new request_t(fname));
    break;

  case FAMEndExist:
    block = false;
    break;

  default:		
    return false;
  }

  if (fsep != NULL) push(fsep);
  return true;
}

void FAMInputComponent::log(FAMEvent *fe)
{
  if (fe == NULL) return;

  CERR(0) << "[fam] " << fe->filename << " ";
  switch (fe->code) {
  case FAMChanged: 	cerr << "changed"; 	break;
  case FAMDeleted: 	cerr << "deleted"; 	break;
  case FAMCreated: 	cerr << "created"; 	break;
  case FAMExists: 	cerr << "exists"; 	break;
  case FAMEndExist:	cerr << "end exist";	break;
  default:		cerr << "? (" << (int) fe->code << ")";
  }
  cerr << endl;
  
}

bool FAMInputComponent::getFileName(FAMEvent *fe, text &t)
{
  if (*(fe->filename) == '/') {
    t.init(fe->filename);
  } else {
    text tmp = monitored_dirs.valAtOrNil((fe->fr).reqnum+1);
    if (tmp.isNull()) return false;
    snprintf(path_buf, PATH_MAX, "%s/%s", tmp.data(), fe->filename);
    t.init(path_buf);
  }
  return true;
}

int FAMInputComponent::getDirName(text &name)
{
  struct stat st;
  if (lstat(name.data(), &st) < 0) {
    pandora_pwarning(name);
    return -1;
  }
  
  if (!S_ISDIR(st.st_mode)) {
    if (!S_ISREG(st.st_mode)) 		return -1;
    if (dirname(name.data()) == NULL) 	return -1;
    name.update();
  }
  return st.st_mode;
}

bool FAMInputComponent::add(Packet *pkt)
{
  if (doStop) {
    cleanPacket(pkt);
    return false;
  }

  locatePacket0(TextValuePacket, vpp, pkt);

  if (vpp == NULL) return_clean(pkt);

  toMonitor.put(new request_t((vpp->val).data()));
  //pandora_debug("[fam] will monitor: " << (vpp->value())->s);
  cleanPacket(pkt);
  return false;
}
