/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "filetourlcomponent.h"
#include <pandora_components/fseventpacket.h>

component_export(FileToURLComponent, FSEventPacket, FSEventPacket);

FileToURLComponent::FileToURLComponent(void)
  : name(NULL), nrules(0)
{
  registerOption("file", &name);
}

bool FileToURLComponent::add(Packet *pkt) 
{
  FSEventPacket *fsep = static_cast<FSEventPacket *>(pkt);

  if ((fsep->path).isNull()) {
    cleanPacket(fsep);
    return false;
  }

  bool found = false;

  for (int i = 0; i < nrules; ++i) {
    pcre_repl_t *pr = mappings[i];
    if (pr->match((fsep->path).data(), (fsep->path).length(), 
		  buffer, sizeof(buffer))) {
      (fsep->path).init(buffer);
      found = true;
    }
  }

  if (found) push(fsep);
  else cleanPacket(fsep);

  return false;
}

void FileToURLComponent::setup(void)
{
  if (name == NULL) return;
  FILE *f = fopen(name, "r");
  if (f == NULL) {
    pandora_warning(name << ": " << strerror(errno));
    return;
  }

  const char *delims = " \t\r\n";

  int lc = 0;
  while (fgets(buffer, sizeof(buffer), f) != NULL) {
    ++lc;
    if (*buffer == '#') continue;
    char *pattern = strtok(buffer, delims);
    if (pattern == NULL) continue;
    char *repl    = strtok(NULL, delims);
    if (repl == NULL) {
      pandora_warning(name << ": invalid format at line " << lc);
      continue;
    }
    pcre_repl_t *pr = new pcre_repl_t(pattern, repl);
    if (pr->isSet()) {
      mappings[nrules] = pr;
      ++nrules;
      if (nrules == max_rules) break;
    } else {
      pandora_warning(pattern << ": failed to compile pattern at line " << lc);
      __DELETE(pr);
    }
  }
  fclose(f);
}

void FileToURLComponent::cleanup(void)
{
  for (int i = 0; i < nrules; ++i) __DELETE(mappings[i]);
  nrules = 0;
}

bool FileToURLComponent::pcre_repl_t::match(const char *str, int length,
					    char *buf, int size) 
{
  pandora_assert(length > 0);
  int ovector[vec_size];
  int n = pcre_exec(pattern, NULL, str, length, 0, 0, 
		    ovector, vec_size);
  if (n <= 0) return false;

  int first = ovector[0], last = ovector[1];
  if (size < first) return false;
  memcpy(buf, str, first);
  size -= first;

  int buf_offset = first, repl_offset = 0, cpy_size = 0;
  int beg , end , num;

  for (int i = 0; i < nsubs; ++i) {
    beg = sub_repl[3*i];
    end = sub_repl[3*i+1];
    num = sub_repl[3*i+2];

    cpy_size = beg - repl_offset;
    if (size < cpy_size) return false;
    memcpy(buf + buf_offset, repl + repl_offset, cpy_size);
    buf_offset += cpy_size;
    repl_offset += cpy_size;
    size -= cpy_size;
    //pandora_debug(i << "a> " << cpy_size << " / " << (repl + repl_offset));

    cpy_size = pcre_copy_substring(str, ovector, n, num, 
				   buf + buf_offset, size);
    if (cpy_size < 0) {
      pandora_warning("pcre_copy_substring #" << num << " failed");
      continue;
    }
    buf_offset += cpy_size;
    repl_offset += (end - beg);
    size -= cpy_size;    
    //pandora_debug(i << "b> " << cpy_size << " / " << (repl + repl_offset));
  }

  cpy_size = (repl_len - repl_offset);
  if (size < cpy_size) return false;  
  memcpy(buf + buf_offset, repl + repl_offset, cpy_size);
  buf_offset += cpy_size;
  size -= cpy_size;    
  
  cpy_size = strlen(buf + buf_offset); // include last '\0'
  if (size < cpy_size) return false;
  memcpy(buf + buf_offset, str + last, cpy_size);
  buf_offset += cpy_size;
  buf[buf_offset] = '\0';
  return true;
}

FileToURLComponent::pcre_repl_t::pcre_repl_t(const char *pat,
					     const char *rep)
  : nsubs(0), pattern(NULL), repl(NULL), repl_len(0)
{
  if ((pat == NULL) | (rep == NULL)) {
    pandora_warning("invalid args: pattern = " << pat << ", repl = " << rep);
    return;
  }

  int tmp;
  const char *errbuf;

  pattern = pcre_compile(pat, 0, &errbuf, &tmp, NULL);
  if (pattern == NULL) {
    pandora_warning(pat << ": " << errbuf);
    return;
  }

  repl = xstrdup(rep);
  repl_compile();
}

FileToURLComponent::pcre_repl_t::~pcre_repl_t(void)
{
  if (pattern != NULL) {
    pcre_free(pattern);
    pattern = NULL;
  }
  __FREE(repl);
}

void FileToURLComponent::pcre_repl_t::repl_compile(void)
{
  if (repl == NULL) return;
  repl_len = strlen(repl);
  pandora_assert(repl_len < 256);
  char *ptr = repl;
  while((ptr = strchr(ptr, '$')) != NULL) {
    if ((ptr != repl) && (*(ptr-1) == '\\')) { 
      ++ptr; 
      continue; 
    }

    char *endptr;
    int val = strtol(ptr+1, &endptr, 10);
    //pandora_debug("found sub expression #" << nsubs << " [" << val << "]");
    if (val <= 0) {
      ++ptr;
      continue;
    }

    sub_repl[3 * nsubs + 0] = ptr - repl;
    sub_repl[3 * nsubs + 1] = endptr - repl;
    sub_repl[3 * nsubs + 2] = val;
    ptr = endptr;
    ++nsubs;
    if (nsubs == max_subs) break;
  }
}
