/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <errno.h>
#include <libpandora/conf/string.h>
#include <libpandora/conf/snprintf.h>
}

#include <iostream>

#include <libpandora/pandora.h>
#include <libpandora/dynloader.h>
#include <libpandora/clock.h>
#include <libpandora/resource_entry.h>
#include <pandora_components/controlpacket.h>
#include <pandora_components/valuepacket.h>
#include "guileinputcomponent.h"
#include "guile_pandora_proc.h"
#include "guile_pandora_types.h"

#define PANDORA_DEF_PORT 23023

typedef SCM (*gfunc)();

static SCM pandora_set_stack_name_x(SCM s_pstack, SCM s_name)
{
  SCM_ASSERT(SCM_PSTACKP(s_pstack), s_pstack, SCM_ARG1, 
	     "set-stack:name");
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_ROSTRINGP(s_name), s_name, SCM_ARG2, 
	     "set-stack:name");

  StackEntry *pstack = (StackEntry *) SCM_CDR(s_pstack);
  scm2text(s_name, pstack->id);
  
  return SCM_UNSPECIFIED;
}


static SCM pandora_get_stack_name(SCM s_pstack)
{
  SCM_ASSERT(SCM_PSTACKP(s_pstack), s_pstack, SCM_ARG1, 
	     "stack:name");

  StackEntry *pstack = (StackEntry *) SCM_CDR(s_pstack);

  return text2scm(pstack->id);
}


static SCM pandora_get_stack_components(SCM s_pstack)
{
  SCM_ASSERT(SCM_PSTACKP(s_pstack), s_pstack, SCM_ARG1, 
	     "stack:components");

  StackEntry *se = (StackEntry *) SCM_CDR(s_pstack);
  SCM comp_list = SCM_LIST0;
  for (int i = se->getNbComps()-1; i >= 0; --i) {
    const CompEntry *ce = se->getCompEntry(i);
    SCM pcomp = ce2scm(ce);
    comp_list = gh_cons(pcomp, comp_list);
  }

  return comp_list;
}

static SCM pandora_set_stack_components_x(SCM s_pstack, SCM s_comp_list)
{
  SCM_ASSERT(SCM_PSTACKP(s_pstack), s_pstack, SCM_ARG1, 
	     "set-stack:components");
  SCM_ASSERT(gh_list_p(s_comp_list), s_comp_list, 
	     SCM_ARG2, "set-stack:components");


  StackEntry *se = (StackEntry *) SCM_CDR(s_pstack);
  se->clearComp();

  SCM s_clist = s_comp_list;

  int stack_index =     gh_length(s_comp_list);
  for (int i = 0; i < stack_index && gh_pair_p(s_clist); ++i) {
    se->pushComponent(*scm2ce(gh_car(s_clist)));
    s_clist = gh_cdr(s_clist);
  }

  return SCM_UNSPECIFIED;
}

static SCM pandora_get_comp_name(SCM s_pcomp)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "component:name");

  CompEntry *pcomp = (CompEntry *) SCM_CDR(s_pcomp);

  return text2scm(pcomp->id);
}

static SCM pandora_set_comp_name_x(SCM s_pcomp, SCM s_name)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "set-component:name");
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_ROSTRINGP(s_name), s_name, SCM_ARG2, 
	     "set-component:name");

  CompEntry *pcomp = (CompEntry *) SCM_CDR(s_pcomp);
  scm2text(s_name, pcomp->id);
  
  return SCM_UNSPECIFIED;
}

static SCM pandora_get_comp_type(SCM s_pcomp)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "component:type");

  CompEntry *ce = (CompEntry *) SCM_CDR(s_pcomp);
  return gh_long2scm(ce->type);
}

static SCM pandora_set_comp_type_x(SCM s_pcomp, SCM s_type)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "set-component:type");
  SCM_ASSERT(SCM_INUMP(s_type), s_type, 
	     SCM_ARG2, "set-component:type");
  
  CompEntry *ce = (CompEntry *) SCM_CDR(s_pcomp);
  ce->type = gh_scm2long(s_type);
  return SCM_UNSPECIFIED;
}

static SCM pandora_comp_is_macro(SCM s_pcomp)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "component:macro?");
  CompEntry *ce = (CompEntry *) SCM_CDR(s_pcomp);
  return SCM_BOOL(ce->ctype == CompEntry::macro);
}

static SCM pandora_get_comp_options(SCM s_pcomp)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "component:options");

  CompEntry *ce = (CompEntry *) SCM_CDR(s_pcomp);
  
  SCM option_list =  SCM_LIST0;
  for (int i = ce->getNbOptions()-1; i >= 0; --i) {
    const OptionEntry *oe = ce->getOptionEntry(i);
    SCM poption = oe2scm(oe);
    option_list = gh_cons(poption, option_list);
  }

  return option_list;
}


static SCM pandora_set_comp_options_x(SCM s_pcomp, SCM s_option_list)
{
  SCM_ASSERT(SCM_PCOMPP(s_pcomp), s_pcomp, SCM_ARG1, 
	     "set-component:options");
  SCM_ASSERT(gh_list_p(s_option_list), s_option_list, 
	     SCM_ARG2, "set-component:options");


  CompEntry *ce = (CompEntry *) SCM_CDR(s_pcomp);
  ce->clearOptions();

  SCM s_olist = s_option_list;
  int option_index = gh_length(s_option_list);
  for (int i = 0; i < option_index && gh_pair_p(s_olist); ++i) {
    ce->pushOption(*scm2oe(gh_car(s_olist)));
    s_olist = gh_cdr(s_olist);
  }
  return SCM_UNSPECIFIED;
}


static SCM pandora_get_option_name(SCM s_poption)
{
  SCM_ASSERT(SCM_POPTIONP(s_poption), s_poption, SCM_ARG1, 
	     "option:name");

  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  return text2scm(poption->id);
}

static SCM pandora_set_option_name_x(SCM s_poption, SCM s_name)
{
  SCM_ASSERT(SCM_POPTIONP(s_poption), s_poption, SCM_ARG1, 
	     "set-option:name");
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_ROSTRINGP(s_name), s_name, SCM_ARG2, 
	     "set-option:name");

  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  scm2text(s_name, poption->id);

  return SCM_UNSPECIFIED;
}

static SCM pandora_get_option_value(SCM s_poption)
{
  SCM_ASSERT(SCM_POPTIONP(s_poption), s_poption, SCM_ARG1, 
	     "option:value");

  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  return mv2scm(poption->mv);
}

static SCM pandora_set_option_value_x(SCM s_poption, SCM s_value)
{
  SCM_ASSERT(SCM_POPTIONP(s_poption), s_poption, SCM_ARG1, 
	     "set-option:value");

  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  int type = pandora_get_value_type(s_value);
  if (type != MultiValue::undefined) {
    scm2mv(s_value, poption->mv);
  } else {
    scm_wrong_type_arg("set-option:value", SCM_ARG2, s_value);
  }
  
  return SCM_UNSPECIFIED;
}

static SCM pandora_connect(SCM s_host, SCM s_port)
{
  char *host = NULL;
  int port = 0;

  SCM_ASSERT(SCM_NIMP(s_host) && SCM_ROSTRINGP(s_host), s_host, 
	     SCM_ARG1, "pandora-connect");
  if (s_port != SCM_UNDEFINED) {
    SCM_ASSERT(SCM_INUMP(s_port), s_port, 
	       SCM_ARG2, "pandora-connect");
    port = SCM_INUM(s_port);
  } else {
    port = PANDORA_DEF_PORT;
  }
  
  host = SCM_ROCHARS(s_host);
  
  return SCM_BOOL(GuileInputComponent::connect(host, port));
}

/****************************\
 * Commands sent to pandora *
\****************************/

static SCM pandora_set_lib(SCM s_lib, SCM s_version, SCM s_loc, SCM s_deps) 
{
  SCM_ASSERT(SCM_NIMP(s_lib) && SCM_ROSTRINGP(s_lib), s_lib, 
	     SCM_ARG1, "pandora-set-library");

  SCM_ASSERT(SCM_INUMP(s_version), s_version, 
	     SCM_ARG2, "pandora-set-library");

  SCM_ASSERT(SCM_NIMP(s_loc) && SCM_ROSTRINGP(s_loc), s_loc, 
	     SCM_ARG3, "pandora-set-library");

  if (s_deps != SCM_UNDEFINED) {
    SCM_ASSERT(SCM_NIMP(s_deps) && SCM_CONSP(s_deps), s_deps, 
	       SCM_ARG4, "pandora-set-library");
  } else {
    s_deps = SCM_LIST0;
  }

  ControlPacket *cp = new ControlPacket(ControlPacket::set_lib);
  text lib = text(SCM_ROCHARS(s_lib));
  cp->writeParam(lib);
  long version = SCM_INUM(s_version);
  cp->writeParam(version);
  text loc = text(SCM_ROCHARS(s_loc));
  cp->writeParam(loc);
  int length = gh_length(s_deps);
  cp->writeParam(length);
  for (int i = 0; i < length; ++i) {
    SCM s_dep = gh_car(s_deps);
    SCM_ASSERT(SCM_NIMP(s_dep) && SCM_ROSTRINGP(s_dep), s_dep, 
	       SCM_ARG3, "pandora-set-library");
    text dep = text(SCM_ROCHARS(s_dep));
    cp->writeParam(dep);
    s_deps = gh_cdr(s_deps);
  }

  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_get_lib(SCM s_libname) 
{
  SCM_ASSERT(SCM_NIMP(s_libname) && SCM_ROSTRINGP(s_libname), s_libname, 
	     SCM_ARG1, "pandora-get-library");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_lib);
  text libname = text(SCM_ROCHARS(s_libname));
  cp->writeParam(libname);

  GuileInputComponent::sendControl(cp);

  int version;
  if (!GuileInputComponent::response(version)) return SCM_BOOL_F;
  
  text loc;
  if (!GuileInputComponent::response(loc)) return SCM_BOOL_F;  

  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    text lib;
    if (!GuileInputComponent::response(lib)) return SCM_BOOL_F;
    s_list = gh_cons(text2scm(lib), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if ((n < 0) | (!ret)) s_list = SCM_BOOL_F;
  return gh_cons(text2scm(loc), gh_cons(SCM_MAKINUM(version), s_list));
}

static SCM pandora_list_libs(void) 
{
   GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::list_libs));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    text lib;
    if (!GuileInputComponent::response(lib)) return SCM_BOOL_F;
    s_list = gh_cons(text2scm(lib), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if (!ret) s_list = SCM_BOOL_F;
  return s_list;
}

static SCM pandora_set_binding(SCM s_compclass, SCM s_libname, SCM s_prefix) 
{
  SCM_ASSERT(SCM_NIMP(s_compclass) && SCM_ROSTRINGP(s_compclass), s_compclass, 
	     SCM_ARG1, "pandora-set-binding");
  SCM_ASSERT(SCM_NIMP(s_libname) && SCM_ROSTRINGP(s_libname), s_libname, 
	     SCM_ARG2, "pandora-set-binding");
  SCM_ASSERT(SCM_NIMP(s_prefix) && SCM_ROSTRINGP(s_prefix), s_prefix, 
	     SCM_ARG3, "pandora-set-binding");

  
  ControlPacket *cp = new ControlPacket(ControlPacket::set_binding);
  text compclass = text(SCM_ROCHARS(s_compclass));
  cp->writeParam(compclass);
  text libname = text(SCM_ROCHARS(s_libname));
  cp->writeParam(libname);
  text prefix = text(SCM_ROCHARS(s_prefix));
  cp->writeParam(prefix);

  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_get_binding(SCM s_compclass) 
{
  SCM_ASSERT(SCM_NIMP(s_compclass) && SCM_ROSTRINGP(s_compclass), s_compclass, 
	     SCM_ARG1, "pandora-get-binding");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_binding);

  text compclass = text(SCM_ROCHARS(s_compclass));
  cp->writeParam(compclass);

  GuileInputComponent::sendControl(cp);
  
  text lib;
  if (!GuileInputComponent::response(lib)) return SCM_BOOL_F;
  if (!GuileInputComponent::getStatus())   return SCM_BOOL_F;
  
  return text2scm(lib);
}


static SCM pandora_list_symbols(void) 
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::list_symbols));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    text sym;
    if (!GuileInputComponent::response(sym)) return SCM_BOOL_F;
    s_list = gh_cons(text2scm(sym), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if (!ret) s_list = SCM_BOOL_F;
  return s_list;
}

static SCM pandora_add_res(SCM s_id, SCM s_uri, SCM s_prio)
{
  SCM_ASSERT(SCM_NIMP(s_id) && SCM_ROSTRINGP(s_id), s_id, 
	     SCM_ARG1, "pandora-add-resource");

  SCM_ASSERT(SCM_NIMP(s_uri) && SCM_ROSTRINGP(s_uri), s_uri, 
	     SCM_ARG2, "pandora-add-resource");

  if (s_prio != SCM_UNDEFINED) {
    SCM_ASSERT(SCM_INUMP(s_prio), s_prio, 
	       SCM_ARG3, "pandora-add-resource");
  } else {
    s_prio = SCM_MAKINUM(1000);
  }

  ControlPacket *cp = new ControlPacket(ControlPacket::add_res);
  long id = string_hash(SCM_ROCHARS(s_id));
  cp->writeParam(id);
  text uri = text(SCM_ROCHARS(s_uri));
  cp->writeParam(uri);
  int prio = SCM_INUM(s_prio);
  cp->writeParam(prio);

  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_del_res(SCM s_id, SCM s_uri)
{
  SCM_ASSERT(SCM_NIMP(s_id) && SCM_ROSTRINGP(s_id), s_id, 
	     SCM_ARG1, "pandora-add-resource");

  SCM_ASSERT(SCM_NIMP(s_uri) && SCM_ROSTRINGP(s_uri), s_uri, 
	     SCM_ARG2, "pandora-delete-resource");

  ControlPacket *cp = new ControlPacket(ControlPacket::del_res);
  long id = string_hash(SCM_ROCHARS(s_id));
  cp->writeParam(id);
  text uri = text(SCM_ROCHARS(s_uri));
  cp->writeParam(uri);

  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_set_res_pri(SCM s_id, SCM s_uri, SCM s_prio)
{
  SCM_ASSERT(SCM_NIMP(s_id) && SCM_ROSTRINGP(s_id), s_id, 
	     SCM_ARG1, "pandora-add-resource");

  SCM_ASSERT(SCM_NIMP(s_uri) && SCM_ROSTRINGP(s_uri), s_uri, 
	     SCM_ARG2, "pandora-set-resource-priority");

  SCM_ASSERT(SCM_INUMP(s_prio), s_prio, 
	     SCM_ARG3, "pandora-set-resource-priority");

  ControlPacket *cp = new ControlPacket(ControlPacket::set_res_pri);
  long id = string_hash(SCM_ROCHARS(s_id));
  cp->writeParam(id);
  text uri = text(SCM_ROCHARS(s_uri));
  cp->writeParam(uri);
  int prio = SCM_INUM(s_prio);
  cp->writeParam(prio);

  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_list_res(SCM s_id)
{
  SCM_ASSERT(SCM_NIMP(s_id) && SCM_ROSTRINGP(s_id), s_id, 
	     SCM_ARG1, "pandora-add-resource");

  ControlPacket *cp = new ControlPacket(ControlPacket::list_res);
  long id = string_hash(SCM_ROCHARS(s_id));
  cp->writeParam(id);
  GuileInputComponent::sendControl(cp);

  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    resource_entry_t re;
    if (!GuileInputComponent::response(re)) return SCM_BOOL_F;
    s_list = gh_cons(gh_cons(text2scm(re.uri),
			     SCM_MAKINUM(re.priority)), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if (!ret) s_list = SCM_BOOL_F;
  return s_list;
}

static SCM pandora_update_res(SCM s_id)
{
  SCM_ASSERT(SCM_NIMP(s_id) && SCM_ROSTRINGP(s_id), s_id, 
	     SCM_ARG1, "pandora-add-resource");

  ControlPacket *cp = new ControlPacket(ControlPacket::update_res);
  long id = string_hash(SCM_ROCHARS(s_id));
  cp->writeParam(id);
  GuileInputComponent::sendControl(cp);

  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_get_comp_prefix(void)
{
  return gh_str02scm(Component::getPrefix());
}

static SCM pandora_get_packet_prefix(void)
{
  return gh_str02scm(Packet::getPrefix());
}

static SCM pandora_stack_start(SCM s_name, SCM s_threaded)
{
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_ROSTRINGP(s_name), s_name, 
	     SCM_ARG1, "pandora-start");

  bool threaded = true;
  if (s_threaded != SCM_UNDEFINED) {
    SCM_ASSERT(SCM_BOOLP(s_threaded), s_threaded, SCM_ARG2, 
	       "pandora-start");
    threaded = SCM_NFALSEP(s_threaded);
  }

  ControlPacket *cp = new ControlPacket(ControlPacket::start);
  text sid(SCM_ROCHARS(s_name));
  cp->writeParam(sid);
  cp->writeParam(threaded);  
  GuileInputComponent::sendControl(cp);
  stack_handle_t h;
  if (!GuileInputComponent::response(h)) return SCM_BOOL_F;
  bool res = GuileInputComponent::getStatus();
  if (!res) return SCM_BOOL_F;
  return gh_ulong2scm(h);
}

static SCM pandora_stack_stop(SCM s_name)
{
  SCM_ASSERT(SCM_INUMP(s_name) || SCM_BIGP(s_name), s_name, 
	     SCM_ARG1, "pandora-stop");

  ControlPacket *cp = new ControlPacket(ControlPacket::stop);
  stack_handle_t h = gh_scm2ulong(s_name);
  cp->writeParam(h);
  GuileInputComponent::sendControl(cp);

  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_stack_suspend(SCM s_name)
{
  SCM_ASSERT(SCM_INUMP(s_name) || SCM_BIGP(s_name), s_name, 
	     SCM_ARG1, "pandora-suspend");

  ControlPacket *cp = new ControlPacket(ControlPacket::suspend);
  stack_handle_t h = gh_scm2ulong(s_name);
  cp->writeParam(h);
  GuileInputComponent::sendControl(cp);

  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_stack_resume(SCM s_name)
{
  SCM_ASSERT(SCM_INUMP(s_name) || SCM_BIGP(s_name), s_name, 
	     SCM_ARG1, "pandora-resume");

  ControlPacket *cp = new ControlPacket(ControlPacket::resume);
  stack_handle_t h = gh_scm2ulong(s_name);
  cp->writeParam(h);
  GuileInputComponent::sendControl(cp);

  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_get_stack(SCM s_name)
{
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_ROSTRINGP(s_name), s_name, SCM_ARG1, 
	     "pandora-get-stack");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_stack);
  text sid(SCM_ROCHARS(s_name));
  cp->writeParam(sid);
  GuileInputComponent::sendControl(cp);
  
  StackEntry se;
  if (!GuileInputComponent::response(se)) return SCM_BOOL_F;

  if (GuileInputComponent::getStatus()) 
    return se2scm(&se);
  return SCM_BOOL_F;
}


static SCM pandora_get_name(SCM s_name)
{
  SCM_ASSERT(SCM_INUMP(s_name) || SCM_BIGP(s_name), s_name, SCM_ARG1, 
	     "pandora-get-name");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_stack_name);
  stack_handle_t h = gh_scm2ulong(s_name);
  cp->writeParam(h);
  GuileInputComponent::sendControl(cp);
  
  text sid;
  if (!GuileInputComponent::response(sid)) return SCM_BOOL_F;

  if (GuileInputComponent::getStatus()) 
    return text2scm(sid);
  return SCM_BOOL_F;
}

static SCM pandora_get_option(SCM s_stack, SCM s_comp, SCM s_option)
{
  SCM_ASSERT(SCM_INUMP(s_stack) || SCM_BIGP(s_stack), s_stack, SCM_ARG1, 
	     "pandora-get-option");
  SCM_ASSERT(SCM_NIMP(s_comp) && SCM_ROSTRINGP(s_comp), s_comp, SCM_ARG2, 
	     "pandora-get-option");
  SCM_ASSERT(SCM_NIMP(s_option) && SCM_ROSTRINGP(s_option), s_option, SCM_ARG3,
	     "pandora-get-option");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_option);
  stack_handle_t h = gh_scm2ulong(s_stack);
  cp->writeParam(h);
  text comp(SCM_ROCHARS(s_comp));
  cp->writeParam(comp);
  text op(SCM_ROCHARS(s_option));
  cp->writeParam(op);

  GuileInputComponent::sendControl(cp);
  
  MultiValue mv;
  if (!GuileInputComponent::response(mv)) return SCM_BOOL_F;

  if (GuileInputComponent::getStatus()) 
    return mv2scm(mv);

  return SCM_BOOL_F;
}

static SCM pandora_query(SCM s_stack, SCM s_comp, SCM s_option)
{
  SCM_ASSERT(SCM_INUMP(s_stack) || SCM_BIGP(s_stack), s_stack, SCM_ARG1, 
	     "pandora-query");
  SCM_ASSERT(SCM_NIMP(s_comp) && SCM_ROSTRINGP(s_comp), s_comp, SCM_ARG2, 
	     "pandora-query");
  SCM_ASSERT(SCM_NIMP(s_option) && SCM_ROSTRINGP(s_option), s_option, SCM_ARG3,
	     "pandora-query");

  ControlPacket *cp = new ControlPacket(ControlPacket::comp_query);
  stack_handle_t h = gh_scm2ulong(s_stack);
  cp->writeParam(h);
  text comp(SCM_ROCHARS(s_comp));
  cp->writeParam(comp);
  text arg(SCM_ROCHARS(s_option));
  cp->writeParam(arg);

  GuileInputComponent::sendControl(cp);
  
  MultiValue mv;
  if (!GuileInputComponent::response(mv)) return SCM_BOOL_F;

  if (GuileInputComponent::getStatus()) 
    return mv2scm(mv);
  return SCM_BOOL_F;
}

static SCM pandora_get_option_default(SCM s_comp, SCM s_option)
{
  SCM_ASSERT(SCM_NIMP(s_comp) && SCM_ROSTRINGP(s_comp), s_comp, SCM_ARG1, 
	     "pandora-get-option-default");
  SCM_ASSERT(SCM_NIMP(s_option) && SCM_ROSTRINGP(s_option), s_option, SCM_ARG2,
	     "pandora-get-option-default");

  ControlPacket *cp = new ControlPacket(ControlPacket::get_option_def);

  text comp(SCM_ROCHARS(s_comp));
  cp->writeParam(comp);
  text op(SCM_ROCHARS(s_option));
  cp->writeParam(op);

  GuileInputComponent::sendControl(cp);
  
  MultiValue mv;
  if (!GuileInputComponent::response(mv)) return SCM_BOOL_F;

  if (GuileInputComponent::getStatus()) 
    return mv2scm(mv);

  return SCM_BOOL_F;
}

static SCM pandora_list_options(SCM s_comp)
{
  SCM_ASSERT(SCM_NIMP(s_comp) && SCM_ROSTRINGP(s_comp), s_comp, SCM_ARG1, 
	     "pandora-list-options");

  ControlPacket *cp = new ControlPacket(ControlPacket::list_options);
  text comp(SCM_ROCHARS(s_comp));
  cp->writeParam(comp);

  GuileInputComponent::sendControl(cp);
  
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    text id;
    if (!GuileInputComponent::response(id)) return SCM_BOOL_F;
    s_list = gh_cons(text2scm(id), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if (!ret) s_list = SCM_BOOL_F;
  return s_list;
}


static SCM pandora_set_stack(SCM s_pstack)
{
  SCM_ASSERT(SCM_PSTACKP(s_pstack), s_pstack, SCM_ARG1, 
	     "pandora-set-stack");
  
  ControlPacket *cp = new ControlPacket(ControlPacket::set_stack);
  cp->writeParam(*scm2se(s_pstack));
  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_set_option(SCM s_stack, SCM s_comp, SCM s_option, 
			      SCM s_val)
{
  SCM_ASSERT(SCM_NIMP(s_comp) && SCM_ROSTRINGP(s_comp), s_comp, SCM_ARG2, 
	     "pandora-set-option");
  SCM_ASSERT(SCM_NIMP(s_option) && SCM_ROSTRINGP(s_option), s_option, SCM_ARG3,
	     "pandora-set-option");

  ControlPacket *cp = NULL;
  if (SCM_NIMP(s_stack) && SCM_ROSTRINGP(s_stack)) {
    cp = new ControlPacket(ControlPacket::set_option);
    text stk(SCM_ROCHARS(s_stack));
    cp->writeParam(stk);
  } else if (SCM_INUMP(s_stack) || SCM_BIGP(s_stack)) {
    cp = new ControlPacket(ControlPacket::set_option_live);
    stack_handle_t h = gh_scm2ulong(s_stack);
    cp->writeParam(h);
  } else {
    scm_wrong_type_arg("pandora-set-option", SCM_ARG1, s_stack);
  }


  text comp(SCM_ROCHARS(s_comp));
  cp->writeParam(comp);
  text op(SCM_ROCHARS(s_option));
  cp->writeParam(op);
  MultiValue mv;
  scm2mv(s_val, mv);
  cp->writeParam(mv);

  GuileInputComponent::sendControl(cp);
 
  return SCM_BOOL(GuileInputComponent::getStatus());  
}

static SCM pandora_list_defined(void)
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::list_defined));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;

  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    text id;
    if (!GuileInputComponent::response(id)) return SCM_BOOL_F;
    s_list = gh_cons(text2scm(id), s_list);
  }
  bool ret = GuileInputComponent::getStatus();
  if (!ret) s_list = SCM_BOOL_F;
  return s_list;
}

static SCM pandora_list_running(void)
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::list_running));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;
  
  SCM s_list = SCM_LIST0;
  for (int i = 0; i < n; ++i) {
    stack_handle_t h;
    if (!GuileInputComponent::response(h)) return SCM_BOOL_F;
    s_list = gh_cons(gh_ulong2scm(h), s_list);
  }

  if (!GuileInputComponent::getStatus()) s_list = SCM_BOOL_F;

  return s_list;
}

static SCM pandora_comp_alive(void) 
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::comp_alive));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;
  if (!GuileInputComponent::getStatus()) return SCM_BOOL_F;

  return SCM_MAKINUM(n);
}

static SCM pandora_packet_alive(void) 
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::packet_alive));
  int n;
  if (!GuileInputComponent::response(n)) return SCM_BOOL_F;
  if (!GuileInputComponent::getStatus()) return SCM_BOOL_F;

  return SCM_MAKINUM(n);
}

static SCM pandora_clean(SCM s_name) 
{
  stack_handle_t h = NIL_STACK_HANDLE;

  if (s_name != SCM_UNDEFINED) {
    SCM_ASSERT(SCM_INUMP(s_name) || SCM_BIGP(s_name), s_name, SCM_ARG1, 
	       "pandora-clean");
    h = gh_scm2ulong(s_name);
  }

  ControlPacket *cp = new ControlPacket(ControlPacket::clean);
  cp->writeParam(h);
  GuileInputComponent::sendControl(cp);
  
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_quit(void) 
{
  GuileInputComponent::sendControl(new ControlPacket
				   (ControlPacket::quit));
  return SCM_BOOL(GuileInputComponent::getStatus());
}

static SCM pandora_start_clock(void) 
{
  wallclock->run();
  return SCM_UNDEFINED;
}

static SCM pandora_set_verbosity(SCM s_level) 
{
  SCM_ASSERT(SCM_INUMP(s_level), s_level, SCM_ARG1, "pandora-set-verbosity");
  int level = SCM_INUM(s_level);
  return SCM_MAKINUM(set_verbosity(level));
}

static SCM _pandora_next_packet(packet_pipe_t *pkt_pipe)
{
  if (pkt_pipe == NULL) return SCM_BOOL_F;

  Packet *pkt = pkt_pipe->get();
  locatePacket0(IntValuePacket, vp, pkt);
  if (vp == NULL) return SCM_BOOL_F;

  SCM value = gh_make_vector(SCM_MAKINUM(3), SCM_BOOL_F);
  gh_vector_set_x(value, SCM_MAKINUM(0),
		  gh_cons(gh_long2scm(vp->timeStamp.tv_sec), 
			  gh_long2scm(vp->timeStamp.tv_usec)));
  if (!(vp->tag).isNull())
    gh_vector_set_x(value, SCM_MAKINUM(1), take_text(vp->tag));
  gh_vector_set_x(value, SCM_MAKINUM(2), SCM_MAKINUM(vp->val));
  
  cleanPacket(pkt);
  return value;
}

static SCM pandora_next_packet(SCM s_ppipe)
{
  SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, "pandora-next-packet");
  packet_pipe_t *pkt_pipe = pandora_get_pipe_ptr(s_ppipe);
  return _pandora_next_packet(pkt_pipe);
}

static SCM pandora_next_packet_ready(SCM s_ppipe)
{
  SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, 
	     "pandora-next-packet-ready");
  packet_pipe_t *pkt_pipe = pandora_get_pipe_ptr(s_ppipe);
  return SCM_BOOL(!pkt_pipe->isEmpty());
}

static SCM pandora_next_packet_nb(SCM s_ppipe)
{
  SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, "pandora-next-packet-nb");
  packet_pipe_t *pkt_pipe = pandora_get_pipe_ptr(s_ppipe);
  if (pkt_pipe->isEmpty()) return SCM_BOOL_F;
  return _pandora_next_packet(pkt_pipe);  
}

static void boot_pandora(void)
{
  pandora_init();
  char stk[2048];
  snprintf(stk, sizeof(stk), 
	   "%%gstack { @GuileInputComponent [$shell = false] "
	   " @ControlComponent }");
  pandora->init(stk);
  pandora->start(text("gstack"), true, true);
}

void scm_init_pguile_module(void)
{
  scm_register_module_xxx("pguile", 
			  (void *)&init_pandora_guile);
}

void init_pandora_guile(void)
{
  if (pandora == NULL) boot_pandora();
  pandora_assert(pandora != NULL);

  pandora_init_types();
  scm_init_error();

  gh_new_procedure1_0("pandora-next-packet",  	  &pandora_next_packet);
  gh_new_procedure1_0("pandora-next-packet-nb",	  &pandora_next_packet_nb);
  gh_new_procedure1_0("pandora-next-packet-ready?",
		      &pandora_next_packet_ready);

  gh_new_procedure1_1("pandora-connect",  	  &pandora_connect);
						 
  gh_new_procedure4_0("pandora-set-library",  	  &pandora_set_lib);
  gh_new_procedure1_0("pandora-get-library",  	  &pandora_get_lib);
  gh_new_procedure0_0("pandora-list-libraries",   &pandora_list_libs);
  gh_new_procedure3_0("pandora-set-binding",  	  &pandora_set_binding);
  gh_new_procedure1_0("pandora-get-binding",  	  &pandora_get_binding);
  gh_new_procedure0_0("pandora-list-symbols",  	  &pandora_list_symbols);
  gh_new_procedure0_0("pandora-get-component-prefix",  	  
		      &pandora_get_comp_prefix);
  gh_new_procedure0_0("pandora-get-packet-prefix",  	  
		      &pandora_get_packet_prefix);

  gh_new_procedure2_1("pandora-add-resource", 	  
		      &pandora_add_res);
  gh_new_procedure2_0("pandora-delete-resource", 	  
		      &pandora_del_res);
  gh_new_procedure3_0("pandora-set-resource-priority",
		      &pandora_set_res_pri);
  gh_new_procedure1_0("pandora-list-resources",	  
		      &pandora_list_res);
  gh_new_procedure1_0("pandora-update-resources",   
		      &pandora_update_res);
						 
  gh_new_procedure1_0("pandora-stop",  	 	  &pandora_stack_stop);
  gh_new_procedure1_0("pandora-get-name",  	  &pandora_get_name);
  gh_new_procedure1_1("pandora-start",  	  &pandora_stack_start);
  gh_new_procedure1_0("pandora-suspend",  	  &pandora_stack_suspend);
  gh_new_procedure1_0("pandora-resume",  	  &pandora_stack_resume);
  gh_new_procedure1_0("pandora-get-stack",  	  &pandora_get_stack);
  gh_new_procedure3_0("pandora-query",  	  &pandora_query);
  gh_new_procedure3_0("pandora-get-option",  	  &pandora_get_option);
  gh_new_procedure2_0("pandora-get-option-default",  	  
		      &pandora_get_option_default);
  gh_new_procedure1_0("pandora-list-options",  	  &pandora_list_options);
  gh_new_procedure1_0("pandora-set-stack",  	  &pandora_set_stack);
  gh_new_procedure0_0("pandora-list-defined", 	  &pandora_list_defined);
  gh_new_procedure0_0("pandora-list-running",  	  &pandora_list_running);
  gh_new_procedure4_0("pandora-set-option",  	  &pandora_set_option);
  gh_new_procedure0_0("pandora-alive-components", &pandora_comp_alive);
  gh_new_procedure0_0("pandora-alive-packets",    &pandora_packet_alive);
						  
  gh_new_procedure0_1("pandora-clean",  	  &pandora_clean);
  gh_new_procedure0_0("pandora-quit",  	 	  &pandora_quit);
  gh_new_procedure0_0("pandora-start-clock", 	  &pandora_start_clock);
  gh_new_procedure1_0("pandora-set-verbosity", 	  &pandora_set_verbosity);
    						  
  gh_new_procedure0_0("make-pandora-stack",  	  &pandora_make_stack);
  gh_new_procedure0_0("make-pandora-component",   &pandora_make_component);
  gh_new_procedure0_0("make-pandora-macro",       &pandora_make_macro);
  gh_new_procedure0_0("make-pandora-option",  	  &pandora_make_option);
  gh_new_procedure1_0("make-pandora-packet-pipe", &pandora_make_packet_pipe);
  gh_new_procedure1_0("pandora-parse",  	  &pandora_parse);
  gh_new_procedure1_0("pandora-print",  	  &pandora_print);

  gh_new_procedure1_0("stack:name", 		  &pandora_get_stack_name);
  gh_new_procedure2_0("set-stack:name", 	  &pandora_set_stack_name_x);
  gh_new_procedure2_0("set-stack:components",     &pandora_set_stack_components_x);
  gh_new_procedure1_0("stack:components", 	  &pandora_get_stack_components);
  gh_new_procedure1_0("component:name", 	  &pandora_get_comp_name);
  gh_new_procedure2_0("set-component:name", 	  &pandora_set_comp_name_x);
  gh_new_procedure1_0("component:type", 	  &pandora_get_comp_type);
  gh_new_procedure2_0("set-component:type", 	  &pandora_set_comp_type_x);
  gh_new_procedure1_0("component:options", 	  &pandora_get_comp_options);
  gh_new_procedure2_0("set-component:options",    &pandora_set_comp_options_x);
  gh_new_procedure1_0("component:macro?", 	  &pandora_comp_is_macro);

  gh_new_procedure1_0("option:name", 		  &pandora_get_option_name);
  gh_new_procedure2_0("set-option:name", 	  &pandora_set_option_name_x);
  gh_new_procedure1_0("option:value", 		  &pandora_get_option_value);
  gh_new_procedure2_0("set-option:value", 	  &pandora_set_option_value_x);
  scm_add_feature("pguile");
  //pandora_debug("[pguile module loaded]");
}
