/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <libpandora/conf/time.h>

#include "httploggercomponent.h"
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/timeval.h>

component_export(HTTPLoggerComponent, HTTPTransPacket+,);

bool HTTPLoggerComponent::add(Packet *pkt) 
{
  locatePacket0(HTTPTransPacket, httptp, pkt);
  locatePacket(HTTPPacket, 	 httpp,  httptp);
  locatePacket(TCPPacket, 	 tcpp,   httpp);
  locatePacket(IPPacket, 	 ipp,    tcpp);

  if (ipp == NULL) {
    discard(pkt);
    return false;
  }
  
  bool matchok = httptp->matchOK();
  bool reqok = httptp->reqOK();
  bool respok = httptp->respOK();  

  ostream *f = (matchok ? matches : (reqok ? reqs : resps));
  
  if (reqok) {
    *f << ipp->src << ':' << ntohs(tcpp->sport) << ' '
       << ipp->dst << ':' << ntohs(tcpp->dport) << ' ';
  } else {
    *f << ipp->dst << ':' << ntohs(tcpp->dport) << ' '
       << ipp->src << ':' << ntohs(tcpp->sport) << ' ';
  }

  *f << "- - "; // I cannot guess remote logname/username

  print_date(f, (httptp->timeStamp).tv_sec);
  print_date(f, ((httptp->timeStamp + httptp->rtt).tv_sec));
  print_date(f, time(NULL));
  print_date(f, ((matchok) ? httptp->date_s : httpp->date));

  *f << '"';

  if (reqok) {
    switch (httpp->method) {
    case HTTPPacket::GET:     *f << "GET " ;     break;
    case HTTPPacket::POST:    *f << "POST " ;    break;
    case HTTPPacket::HEAD:    *f << "HEAD " ;    break;
    case HTTPPacket::PUT:     *f << "PUT " ;     break;
    case HTTPPacket::OPTIONS: *f << "OPTIONS " ; break;
    case HTTPPacket::TRACE:   *f << "TRACE " ;   break;
    case HTTPPacket::DELETE:  *f << "DELETE " ;  break;
    case HTTPPacket::UNDEF: default:             break;
    }
  
    int maj = httpp->version / 1000;
    int min = httpp->version - (1000 * maj);
    char *url = strchr((httpp->url).data() + 7, '/');
    if (url == NULL) url = (httpp->url).data();

    *f << url
       << " HTTP/" << maj << '.' << min;
  } else {
    *f << '-';
  }
  *f << "\" ";

  if (respok) {
    *f << httpp->code << ' '
       << httptp->respmlen << ' ';
  } else {
    *f << "- - ";
  }

  *f << '"'
     << httpp->ref << "\" \"" << httpp->useragent << "\" \""
     << httpp->cookie << "\" " << httpp->host << '\n';

  discard(pkt);
  return false;
}  

void HTTPLoggerComponent::setup(void)
{
  if ((reqs == &cerr)  && (matches != &cout)) reqs = matches;
  if ((resps == &cerr) && (reqs != &cerr))    resps = reqs;
}

void HTTPLoggerComponent::print_date(ostream *f, time_t date)
{
  if (date > 0) {
    struct tm *then = localtime(&date);
    strftime(timefmt, sizeof(timefmt), "[%d/%b/%Y:%H:%M:%S %z] ", then);
    *f << timefmt;
  } else {
    *f << "[??""/??""/??:??:??:?? +????] ";
  }
}
