/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>

#include "httpmatchcomponent.h"
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/error.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>

component_export(HTTPMatchComponent, HTTPPacket, HTTPTransPacket);

bool HTTPMatchComponent::add(Packet* pkt)
{
  HTTPPacket *httpp = static_cast<HTTPPacket *>(pkt);

  if (httpp->type == HTTPPacket::unknown) {
    pandora_warning("unrecognized HTTP message!");    
    return_clean(pkt);
  }

  int idx = (int)(httpp->type == HTTPPacket::response);

  list_t *list0=&reqlist[idx];
  list_t *list1=&reqlist[1-idx];
  HTTPTransPacket *match = NULL;
  HTTPPacket *req = NULL, *resp = NULL;
  list_ptr_t ptr;
  bool nogap = false;

  // handle some special response code
  switch(httpp->code) {
  case 100: unmatched(httpp); return false;
  default:  break;
  }

  if (list1->empty()) goto store;

  nogap = (!httpp->gap & !list1->front()->gap);
  if (idx) resp  = httpp; 
  else     req = httpp;

  for (ptr = list1->begin(); ptr != list1->end(); ++ptr) {
    if (idx) req  = *ptr;
    else     resp = *ptr;

    if (check(req, resp, nogap)) goto found;
    nogap = false;
  }
  
  // no match found => store the request at the _end_ of the list
 store:
  list0->push_back(httpp);
  return false;
  
 found:
  pandora_assert(req != NULL);
  pandora_assert(resp != NULL);
  push(new HTTPTransPacket(req, resp));
  list1->erase(ptr);
  return false;
}

bool HTTPMatchComponent::check(HTTPPacket *request, HTTPPacket *response, 
			       bool nogap)
{
  if (response->timeStamp < request->last) return false;
  if (nogap) return true;

  locatePacket(TCPPacket, treq,  request);
  locatePacket(TCPPacket, tresp, response);
  if (treq->ack  == response->beg)  return true;
  // if (treq->ack  >  response->beg) return false;
  return false;
}

void HTTPMatchComponent::flush(list_t *l)
{
  HTTPTransPacket *match = NULL;

  while (l->size() > 0) {
    HTTPPacket *httpp = l->front();
    pandora_assert(httpp != NULL);
    l->pop_front();
    unmatched(httpp);
  }

  pandora_assert(l->empty());
}

void HTTPMatchComponent::unmatched(HTTPPacket *httpp)
{
  if (!strict) {
    push(new HTTPTransPacket(httpp));
  } else {
    cleanPacket(httpp);
  }
}

void HTTPMatchComponent::cleanup(void)
{
#if 0
  if (!reqlist[0].empty() || !reqlist[0].empty())
    pandora_debug("being flushed");
#endif
  flush(&reqlist[0]);
  flush(&reqlist[1]);
}
