/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdlib.h>
#include <libpandora/conf/snprintf.h>
}

#include <pandora_components/httppacket.h>
#include <pandora_components/valuepacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/urlutil.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>

packet_export(HTTPPacket, TextValuePacket|TCPPacket);

HTTPPacket::HTTPPacket(TCPPacket *tcpp) 
  : type(unknown), method(UNDEF), code(0), 
    beg(0), endh(0), endm(0),
    version(0), date((time_t) -1), ce(-1), ct(-1), cl(-1), 
    pragma(unset), ims(-1), ius(-1), lm(-1), expires(-1), 
    cookies(0), via(0), coll(0), 
    dynamic(false), hashed(false), absolute(false), cacheable(false), 
    gap(false), eof(false), hdrOK(false)
{
  timeStamp.tv_sec = last.tv_sec = (time_t) 0;
  if (tcpp == NULL) return;
  packetSetUp(tcpp);
}


HTTPPacket::HTTPPacket(TextValuePacket *vp) 
  : type(unknown), method(UNDEF), hashed(false)
{
  if (vp == NULL) return;
  type = value;
  method = GET;
  url.init((vp->val).data());
  timeStamp = vp->timeStamp;
  cleanPacket(vp);
}


HTTPPacket::HTTPPacket(const HTTPPacket& x) : 
  Packet(x), type(x.type), method(x.method), code(x.code), 
  beg(x.beg), endh(x.endh), endm(x.endm), last(x.last), 
  version(x.version), date(x.date), 
  ce(x.ce), ct(x.ct), cl(x.cl), pragma(x.pragma), 
  ims(x.ims), ius(x.ius), lm(x.lm), expires(x.expires),
  cookies(x.cookies), via(x.via), url(x.url), ref(x.ref), host(x.host), 
  useragent(x.useragent), cookie(x.cookie),
  coll(x.coll), cctrl(x.cctrl), dynamic(x.dynamic), 
  hashed(x.hashed), absolute(x.absolute), cacheable(x.cacheable), 
  gap(x.gap), eof(x.eof), hdrOK(x.hdrOK)
{
}

HTTPPacket& HTTPPacket::operator= (const HTTPPacket& x) 
{
  Packet::operator=(x);
  type = x.type; method = x.method; code = x.code; 
  beg = x.beg; endh = x.endh; endm = x.endm; last = x.last; 
  version = x.version; date = x.date; 
  ce = x.ce; ct = x.ct; cl = x.cl; pragma = x.pragma; 
  ims = x.ims; ius = x.ius; lm = x.lm; expires = x.expires; 
  cookies = x.cookies; via = x.via; url = x.url; ref = x.ref; host = x.host; 
  useragent = x.useragent; cookie = x.cookie;
  coll = x.coll; cctrl = x.cctrl; dynamic = x.dynamic; 
  hashed = x.hashed; absolute = x.absolute; cacheable = x.cacheable; 
  gap = x.gap; eof = x.eof; hdrOK = x.hdrOK; 
  return *this; 
}


void HTTPPacket::print(ostream *f)
{
  locatePacket(TCPPacket, tcpp, this);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL)  return;

  *f << timeStamp << "\t[http] ";
  *f << ipp->src << ':' << ntohs(tcpp->sport) << ' ' 
     << ipp->dst << ':' << ntohs(tcpp->dport) << ' ';

  switch(type) {
  case response:
    *f << "resp " << code << ' ';
    break;
  case request:
    *f << "req  " << (int)method << ' ';
    /* fall through */
  case value:
    if (!hashed) *f << url;
    else *f << (u_short)((url.data())[0]<<8 + (url.data())[1]) << "-(hashed)";
    *f << ' '; 
    break;
  default:
    break;
  }
  *f << version << ' ';
  //*f << SEQ_SUB(endm, endh) << " / " << cl << ' ';
  *f << "[" << beg << ' ' << endm <<  ' ' << tcpp->ack << "] ";

  if (type != value) {
    *f << (gap ? 'G' : '-') << ' ';
  }

  *f << endl;
}

void HTTPPacket::urlCanonicalize(void)
{
  locatePacket(TCPPacket, tcpp, this);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL || url.isNull()) return;

  if (!HTURL_isAbsolute(url.data())) {
    char *url2;
    if (host.isNull()) {
      host.init(22);
      snprintf(host.data(), 22, "%s", intoa((u_int32_t)ipp->dst.s_addr));
      host.update();
    } else {
      char *tmp = strchr(host.data(), ':');
      if (tmp != NULL) *tmp = '\0';
    }
    
    size_t u2len = url.length() + host.length() + 32;
    url2 = (char *) xmalloc(u2len*sizeof(char));
    snprintf(url2, u2len, "http://%s:%d%s", host.data(), ntohs(tcpp->dport),
	     (((url.data())[0] == '*')  ? "/" : url.data()));
    url.take(url2);
  } else {
    absolute = true;
  }

  char *udata = url.data();
  url_canonicalize(&udata);
  url.update();

  if (!ref.isNull()) {
    char *rdata = ref.data();
    url_canonicalize(&rdata);
    ref.update();
  }
  
}

bool HTTPPacket::reject(void) const
{
  return (type == request 
	  && (! ((url.length() > 5) 
		 && (hashed 
		     || (strncmp(url.data(), "http", 4)==0)))));
}

void HTTPPacket::checkDynamicity(void)
{
  if (type == request) {
    if (url.length() < 8) return;
    dynamic |= (strstr(url.data(), "cgi-bin") != NULL);
    dynamic |= (strchr(url.data(), '?') != NULL);
  } else {
    dynamic |= (lm > 0 && lm == date);
    dynamic |= (expires > 0 && expires == date);
  }
}

size_t HTTPPacket::write(char *str, size_t maxlen, int)
{
  size_t count = 0;

  serialVar(type);
  serialVar(method);
  serialVar(code);
  serialVar(beg);
  serialVar(endh);
  serialVar(endm);
  serialVar(last);

  serialVar(version);
  serialVar(date);
  serialVar(ce);
  serialVar(ct);
  serialVar(cl);
  serialVar(pragma);
  serialVar(ims);
  serialVar(ius);
  serialVar(lm);
  serialVar(expires);
  serialVar(cookies);
  serialVar(via);
  serialVar(url);
  serialVar(ref);
  serialVar(host);
  serialVar(useragent);
  serialVar(cookie);
  serialVar(coll);
  //serialVar(cctrl);
  
  serialVar(dynamic);
  serialVar(hashed); 
  serialVar(absolute);
  serialVar(cacheable);
  serialVar(gap);
  serialVar(eof);

  return count;
}

size_t HTTPPacket::read(const char *str, int)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(method);
  unserialVar(code);
  unserialVar(beg);
  unserialVar(endh);
  unserialVar(endm);
  unserialVar(last);
  
  unserialVar(version);
  unserialVar(date);
  unserialVar(ce);
  unserialVar(ct);
  unserialVar(cl);
  unserialVar(pragma);
  unserialVar(ims);
  unserialVar(ius);
  unserialVar(lm);
  unserialVar(expires);
  unserialVar(cookies);
  unserialVar(via);
  unserialVar(url);
  unserialVar(ref);
  unserialVar(host);
  unserialVar(useragent);
  unserialVar(cookie);
  unserialVar(coll);
  //unserialVar(cctrl);
  
  unserialVar(dynamic);
  unserialVar(hashed); 
  unserialVar(absolute);
  unserialVar(cacheable);
  unserialVar(gap);
  unserialVar(eof);
  
  return count;
} 

extern_pandora(algo, bool, httpurl, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(HTTPPacket, httpp, pkt);
  if (httpp == NULL) return false;
  k->set(httpp->url);
  return true;
}

