/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <arpa/inet.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <libpandora/conf/string.h>
#include <libpandora/extract.h>
	   }

#include <iostream>
#include <iomanip>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>

packet_export(IPPacket,);

void IPPacket::init(const u_char *bp, u_int len, const timeval &ts) 
{
  if (sizeof(struct ip) > len) {
    pandora_warning("truncated IP (header)");
    return;
  }
  const ip *iphdr = (const ip *)bp;

  u_int hlen = IP_HL(iphdr) * 4;
  if (hlen > len) {
    pandora_warning("truncated IP (options)");
    hlen = len;
  }

  length = ntohs(iphdr->ip_len);  

  // it looks like there is a minimum capture length around 40 bytes :(
  if (length < len) len = length;
  
#if 0
  if (length > len) {
    pandora_warning("truncated IP (length = " << length 
		    << " > len = " << len << ")");
    return;
  }
#endif
  
#if 0
  _data.setNoCopy((char *) bp, len); // or length??
#else
  _data.set((char *) bp, len);
#endif

  length -= hlen;  
  _data.move(hlen);
  
  ident = 	iphdr->ip_id;
  offset = 	ntohs(iphdr->ip_off);
  ttl = 	iphdr->ip_ttl;
  proto = 	iphdr->ip_p;
  src.s_addr = 	iphdr->ip_src.s_addr;
  dst.s_addr = 	iphdr->ip_dst.s_addr;
  timeStamp = 	ts;
}

IPPacket::IPPacket(const IPPacket& x) 
  : Packet(x), src(x.src), dst(x.dst), ttl(x.ttl), group(x.group), 
  length(0), offset(0), flags(x.flags), 
  proto(x.proto), ident(x.ident), end(0)
{ 
}

IPPacket& IPPacket::operator= (const IPPacket& x) 
{
  Packet::operator=(x);
  src = x.src; dst = x.dst; ttl = x.ttl; group = x.group; 
  length = 0; offset = 0;  flags = x.flags; 
  proto = x.proto; ident = x.ident; end = 0; 
  return *this;
}

void IPPacket::print(ostream *f)
{
  *f << timeStamp << '\t'
     << "[ip] " << src << ' ' << dst << ' '
     << '[' << ntohs(ident) << "] "
     << length << '/' << _data.getLength() << ' ';


  if ( offset & 0x3fff ) {
    *f << "frag " << ntohs(ident) << ':' << length 
       << '@' << (offset & 0x1fff) * 8
       << ((offset & IP_MF)? "+" : "") << ' ';
  }
  
#if 1
  *f << endl;
#else
  *f << "\r"; f->flush();
#endif
}

size_t IPPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(src);
  serialVar(dst);
  serialVar(ttl);
  serialVar(group);  

  beginLevel(0);
  serialVar(length);
  serialVar(flags);
  serialVar(proto);
  serialVar(ident);
  serialVar(_data);

  return count;
}

size_t IPPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(src);
  unserialVar(dst);
  unserialVar(ttl);
  unserialVar(group);  

  beginLevel(0);
  unserialVar(length);
  unserialVar(flags);
  unserialVar(proto);
  unserialVar(ident);
  unserialVar(_data);

  return count;
}

extern_pandora(algo, bool, ipsource, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(IPPacket, ipp, pkt);
  if (ipp == NULL) return false;

  k->set((ipp->src).s_addr);
  
  return true;
}

extern_pandora(algo, bool, ipdest, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(IPPacket, ipp, pkt);
  if (ipp == NULL) return false;

  k->set((ipp->dst).s_addr);

  return true;
}

extern_pandora(algo, bool, ippair, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(IPPacket, ipp, pkt);
  if (ipp == NULL) return false;

  k->set((ipp->src).s_addr, (ipp->dst).s_addr);
  
  return true;
}

extern_pandora(algo, bool, ipcnx, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(IPPacket, ipp, pkt);
  if (ipp == NULL) return false;

  if ((ipp->src).s_addr < (ipp->dst).s_addr) 
    k->set((ipp->src).s_addr, (ipp->dst).s_addr);
  else 
    k->set((ipp->dst).s_addr, (ipp->src).s_addr);
  
  return true;
}

extern_pandora(algo, bool, ipquad, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(IPPacket, ipp, pkt);
  if (ipp == NULL) return false;

  k->set((ipp->src).s_addr, (ipp->dst).s_addr, ipp->ident, ipp->proto);
  
  return true;
}

