/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h> 

extern "C" {
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <stdlib.h>
#include <libpandora/conf/string.h>
#include <assert.h>
#include <errno.h>
 	   }

#include <iostream>
#include <list>

#include <pandora_components/ippacket.h>
#include "ipreasscomponent.h"

component_export(IPReassComponent, IPPacket, IPPacket);

bool IPReassComponent::add(Packet *obj)
{
  IPPacket *ipp = (IPPacket *)obj;

  if ((ipp->_data).isEmpty()) {
    pandora_warning("invalid IP packet, skipping");
    __DELETE(ipp);
    return false;
  }


  if ( !(ipp->offset & 0x3fff) ) {
    pandora_warning("this packet is not fragmented!");
    push(ipp);
    return true;
  }

  PListIter prev, next, tmp, tmp2;

#ifndef IP_OFFSET
#define IP_OFFSET 0x1fff
#endif
  
  ipp->flags = ipp->offset & ~IP_OFFSET;
  ipp->offset &= IP_OFFSET;
  ipp->offset <<= 3;
  ipp->end = ipp->offset + ipp->length;

  if (ipp->end > 65535) {
    pandora_warning("attempt to construct an oversized packet");
    __DELETE(ipp);
    return true;
  }

  if((ipp->flags & IP_MF) == 0) length = ipp->end;
  
  /*
   * Find out which fragments are in front and at the back of us
   * in the chain of fragments so far.  We must know where to put
   * this fragment, right?
   */
  
  prev=plist.end();
  if(!plist.empty()){
    for(next=plist.begin(); next!=plist.end(); ++next){

      pandora_assert((*next) != NULL);

      if((*next)->offset >= ipp->offset) break;
      prev=next;
    }
  }
  else next=plist.end();


  /*
   * We found where to put this one.
   * Check for overlap with preceding fragment, and, if needed,
   * align things so that any overlaps are eliminated.
   */

  if ((prev != plist.end()) && (ipp->offset < (*prev)->end)) {
    u_int i = (*prev)->end - ipp->offset;
    ipp->length -= i;
    ipp->offset += i;
    (ipp->_data).move(i);
  }

  for (tmp=next; tmp != plist.end(); tmp = tmp2) {
    IPPacket *ipp2 = *tmp;
    tmp2 = tmp;
    ++tmp2;
    
    if (ipp2->offset >= ipp->end)
      break;          /* no overlaps at all */

    u_int i = ipp->end - (*next)->offset; /* overlap is 'i' bytes */
    if (ipp2->length > i) {
      ipp2->length -= i;                   /* so reduce size of    */
      ipp2->offset += i;                   /* next fragment        */
      (ipp2->_data).move(i);
    } else {
      ipp2->length = 0;
    }
    
    /*
     *      If we get a frag size of 0, remove the packet
     *      that it goes with.
     */
    if (ipp2->length == 0) {
      next=tmp2;
      plist.erase(tmp);
      __DELETE(ipp2);
    }
  }

  plist.insert(next,ipp);
  
  if (done()) {
    dispatch();
    return true;
  }

  return false;
}

bool IPReassComponent::done(void)
{
  if (length == 0) return false;
  PListIter ptr;
  u_int offset = 0;
  for (ptr=plist.begin(); ptr!=plist.end(); ++ptr) {
    IPPacket *ipp = *ptr;

    pandora_assert(ipp != NULL);
    
    if (ipp->offset > offset) return false;
    
    pandora_assert (ipp->offset == offset);
    
    offset = ipp->end;
  }
  
  pandora_assert ((int)offset == length);
  
  return true;
}


void IPReassComponent::dispatch(void)
{
  pandora_assert (!plist.empty());
  pandora_assert (length < 65536);
  
  IPPacket *ipp = new IPPacket(*(plist.front()));

#if 0
  cerr << "* " << ipp->timeStamp << '\t'
       << ipp->src << ' ' <<ipp->dst << ' '
       << '(' << length << ") "
       << "completed" << endl;
#endif

  (ipp->_data).setEmpty(length);
  ipp->length = length;

  PListIter ptr;
  u_int count = 0;
  for (ptr=plist.begin(); ptr!=plist.end(); ++ptr) {
    IPPacket *ipp2 = *ptr;

    pandora_assert(ipp2 != NULL);
    (ipp->_data).atWrite(ipp2->data(), ipp2->dlength(), ipp2->offset);
    count += ipp2->length;
    
    cleanPacket(ipp2);
  }

  plist.clear();
  
  push(ipp);
}



void IPReassComponent::cleanup(void)
{
  //if (!plist.empty()) dispatch();
  flush();
  length = 0;
}
