/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "multiinputcomponent.h"
#include <libpandora/component.h>
#include <libpandora/inputcomponent.h>
#include <libpandora/select.h>

component_export(MultiInputComponent,,);

MultiInputComponent::MultiInputComponent(void) 
  : nb_succs(0), nb_thr_succs(0), iselect(NULL) 
{ 
  memset((char *)next_inputs, 0, MAX_BRANCHES * sizeof(InputComponent *));
  memset((char *)next_thr_inputs, 0, MAX_BRANCHES * sizeof(InputComponent *));
}

static bool input_process(int fd, void *ptr)
{
  if (ptr == NULL) return true;
  InputComponent *icomp = static_cast<InputComponent *>(ptr);
  return (icomp->process() || (!icomp->isActive()));
}

static void input_finish(int fd, void *ptr)
{
  if (ptr == NULL) return;
  InputComponent *icomp = static_cast<InputComponent *>(ptr);
  //pandora_debug("finished comp [" << icomp << "] #" << icomp->ind);
  (icomp->finish());
  Component::clean(icomp);
}

int MultiInputComponent::init(void) 
{
  for (int i = 0; i < MAX_BRANCHES; ++i) {
    Component *comp = setUpBranch(i+1);
    if (comp == NULL) break;
    if (!comp->input) {
      pandora_warning("successors must be input components");
      clean(comp);
      continue;
    }
    next_inputs[nb_succs] = static_cast<InputComponent *>(comp);
    //pandora_debug("found succesor #" << nb_succs+1);
    ++nb_succs;
  }

  iselect = new Select();
  bool ok = false;

  for (int b = 0; b < nb_succs; ++b) {
    int fd = (next_inputs[b])->init();

    if (fd == ERROR_FILENO) {
      pandora_warning("succesor #" << b << " cannot be started (error)");
      continue;
    }

    ok = true;

    (next_inputs[b])->in_init();

    if (fd < 0) {
      next_thr_inputs[nb_thr_succs] = next_inputs[b];
      ++nb_thr_succs;
    } else {
      iselect->registerFd(fd, (void *)(next_inputs[b]),
			  &input_process, &input_finish);
    }
  }

  if (!ok) {
    __DELETE(iselect);
    return ERROR_FILENO;
  }

  if (nb_thr_succs > 0) iselect->setTimeout(0);

  return THREADED_FILENO;
}

bool MultiInputComponent::process(void) 
{
  for (int i = 0; i < nb_thr_succs; ++i) {
    InputComponent *ic = next_thr_inputs[i];
    bool ret = ic->process();
    if (ret || (!ic->isActive())) {
      //pandora_debug("finished comp [" << ic << "] #" << ic->ind);
      join_thr_input(ic, i);
      --i;
    }
  }

  if (iselect == NULL) return true;
  iselect->select();
  if (iselect == NULL) return true;
  return ((nb_thr_succs == 0) && (iselect->size() == 0));
}

void MultiInputComponent::finish(void) 
{
  for (int i = 0; i < nb_thr_succs; ++i) {
    (next_thr_inputs[i])->finish();
  }  
  __DELETE(iselect);
}

void MultiInputComponent::halt(void)
{
  for (int b = 0; b < nb_succs; ++b) {
    InputComponent *icomp = next_inputs[b];
    if (icomp == NULL || icomp->flushed) continue;
    icomp->halt();
  }
  __DELETE(iselect);
}

void MultiInputComponent::join_thr_input(InputComponent *ic, int rank)
{
  if (ic == NULL) return;
  ic->finish();
  clean(ic);
  if (rank < (nb_thr_succs - 1))
    memmove((void *)(next_thr_inputs + rank),
	    (void *)(next_thr_inputs + rank + 1),
	    (nb_thr_succs - rank - 1) * sizeof(InputComponent *));
  --nb_thr_succs;
  for (int i = 0; i < nb_succs; ++i) 
    if (next_inputs[i] == ic) next_inputs[i] = NULL;
  if (nb_thr_succs == 0 && iselect != NULL) iselect->setTimeout(-1);
}
