/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>
#include <iostream>

extern "C" {
#ifdef __STDC__
struct rtentry;
struct mbuf;
#endif

#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef solaris
#define map __buggy_sol_map__
#endif

# include <net/if.h>
#include <libpandora/conf/if_ether.h>

#ifdef solaris
#undef map
#endif

}

#include "pcaphandlercomponent.h"

#include <pandora_components/ippacket.h>
#include <libpandora/component.h>
#include <iomanip>
#include "machdep.h"

component_export(PcapHandlerComponent,, IPPacket);

Mutex PcapHandlerComponent::mx;

printer PcapHandlerComponent::printers[] = {
  { ether_if_print,	DLT_EN10MB },
  { ether_if_print,	DLT_IEEE802 },
  { null_if_print,	DLT_NULL },
  { ppp_if_print,	DLT_PPP},
  { ppp_bsdos_if_print,	DLT_PPP_BSDOS},
  { raw_if_print,	DLT_RAW},
#ifdef DLT_LINUX_SLL
  { sll_if_print,	DLT_LINUX_SLL },
#endif
  { NULL,               -1 },
};


PcapHandlerComponent::PcapHandlerComponent(void) 
  : recv(0), drops(0),
    pd(0), handler(NULL), pcount(10), userdata(NULL),
    cmdbuf(NULL), device(NULL), RFileName(NULL),
    snaplen(65535), promisc(1), to_ms(500), locked(false),
    ipp(NULL)
{
  registerOption("pcount", &pcount);
  registerOption("filter", &cmdbuf);
  registerOption("device", &device);
  registerOption("file", &RFileName);
  registerOption("snaplen", &snaplen);
}

void PcapHandlerComponent::ether_if_print(u_char *user, const pcap_pkthdr *h,
					  const u_char *p)
{
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;
  u_int len = h->len;
  struct ether_header *ep;

  ep = (struct ether_header *)p;
  
  if (ntohs(ep->ether_type) != ETHERTYPE_IP) {
    /* ether_type not IP, not supported! skip it */
    return;
  }

  me->dispatch(p + sizeof(struct ether_header), 
	       len - sizeof(struct ether_header), 
	       h->ts);
}

#ifdef DLT_LINUX_SLL
#include "sll.h"

void PcapHandlerComponent::sll_if_print(u_char *user, const pcap_pkthdr *h,
					const u_char *p)
{
  //pandora_debug("[sll] packet received");
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;

  u_int len = h->len;
  const struct sll_header *sllp;

  sllp = (const struct sll_header *)p;
  if (ntohs(sllp->sll_protocol) != ETHERTYPE_IP) {
    //pandora_debug("[sll] not IP: #" << ntohs(sllp->sll_protocol));
    /* ether_type not IP, not supported! skip it */
    return;
  }

  me->dispatch(p + SLL_HDR_LEN, len - SLL_HDR_LEN, h->ts);
}

#endif

#define	NULL_HDRLEN 4

void PcapHandlerComponent::null_if_print(u_char *user, const  pcap_pkthdr *h,
				const u_char *p)
{
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;
  pandora_assert(me != NULL);
  u_int len = h->len;
  me->dispatch(p + NULL_HDRLEN, len - NULL_HDRLEN, h->ts);
}

#define	PPP_HDRLEN 4

void PcapHandlerComponent::ppp_if_print(u_char *user, const  pcap_pkthdr *h,
			       const u_char *p)
{
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;
  u_int len = h->len;
  me->dispatch(p + PPP_HDRLEN, len - PPP_HDRLEN, h->ts);
}

void PcapHandlerComponent::ppp_bsdos_if_print(u_char *user, 
					      const  pcap_pkthdr *h,
					      const u_char *p)
{
  pandora_error("PPP BSDOS not implemented!");
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;
  u_int len = h->len;
  me->dispatch(p + PPP_HDRLEN, len - PPP_HDRLEN, h->ts);
}

void PcapHandlerComponent::raw_if_print(u_char *user, const  pcap_pkthdr *h,
			       const u_char *p)
{
  PcapHandlerComponent *me = (PcapHandlerComponent *)user;
  u_int len = h->len;
  me->dispatch(p, len, h->ts);
}


pcap_handler PcapHandlerComponent::lookup(int type) 
{
  //pandora_debug("looking type: #" <<  type);

  for (printer *p = printers; p->f != NULL; ++p) {
    if (type == p->type)
      return p->f;
  }

  pandora_warning("unknown data link type " << type);
  return (pcap_handler) 0;
}

int PcapHandlerComponent::init(void)
{
  if (pd != 0) return ERROR_FILENO;

  bpf_u_int32 localnet, netmask;
  struct bpf_program fcode;
  char *ebuf;

  mx.lock(); locked = true;
  ebuf=(char *)xmalloc(PCAP_ERRBUF_SIZE*sizeof(char));

  if (abort_on_misalignment(ebuf, PCAP_ERRBUF_SIZE) < 0){
    pandora_warning(ebuf);
    goto finished;
  }

  if (RFileName != NULL) {
    setuid(getuid());

    pd = pcap_open_offline(RFileName, ebuf);
    if (pd == 0){
      pandora_warning(ebuf);
      goto finished;
    }

    localnet = 0;
    netmask = 0;
  }  else {
    if (device == NULL) {
      device = pcap_lookupdev(ebuf);
      if (device == NULL){
	pandora_warning(ebuf);
	goto finished;
      }
    }
    //pandora_debug("snaplen: " << snaplen);
    pd = pcap_open_live(device, snaplen, promisc, to_ms, ebuf);
    
    if (pd == 0) {
      pandora_warning(ebuf);
      goto finished;
    }
    
    if (pcap_lookupnet(device, &localnet, &netmask, ebuf) < 0) {
      localnet = 0;
      netmask = 0;
      pandora_warning(ebuf);
    }
    /*
     * Let user own process after socket has been opened.
     */
    setuid(getuid());
  }
  
  if (cmdbuf != NULL) {
    //pandora_debug("setting filter: '" << cmdbuf << "'");
    if (pcap_compile(pd, &fcode, cmdbuf, 1, netmask) < 0){
      pandora_warning(pcap_geterr(pd));
      pcap_close(pd); pd = 0; goto finished;
    }
    
    if (pcap_setfilter(pd, &fcode) < 0){
      pandora_warning(pcap_geterr(pd));
      pcap_close(pd); pd = 0; goto finished;
    }
  }
  
  handler = lookup(pcap_datalink(pd));
  //pandora_debug(pcap_datalink(pd) << " " << (void *)handler);

  userdata = (u_char *) this;
  if (handler == NULL) { pcap_close(pd); pd = 0; }
  
 finished:  
  xfree(ebuf);

  /* never use a "pcount" of 0 */
  if (pcount == 0) pcount = -1;
  
  locked = false;
  mx.unlock();

  updateStats();

  return (pd != 0) ? pcap_fileno(pd) : ERROR_FILENO;
}

bool PcapHandlerComponent::process(void)
{
  int status = pcap_dispatch(pd, pcount, handler, userdata);

  if (status <= 0) {
    if (status < 0) pandora_warning(pcap_geterr(pd));
    return (RFileName != NULL);
  }
  
  return false;
}

void PcapHandlerComponent::finish(void) 
{ 
  if (updateStats() && (drops > 0)) {
    pandora_warning("[pcap] received: " << recv);
    pandora_warning("[pcap] dropped:  " << drops);
    pandora_warning("[pcap] rate:     " 
		    << 100.0*(float)drops / (float)(drops + recv) << "%");
  }
  
  mx.tryLock();
  if (pd != 0) {
    pcap_close(pd);
    pd = 0;
  }
  locked = false;
  mx.unlock();
  resetStats();
}

bool PcapHandlerComponent::updateStats(void)
{
  if (pd == 0 || pcap_file(pd) != NULL) return false;

  struct pcap_stat stat;
  if (pcap_stats(pd, &stat) < 0) {
    pandora_warning(pcap_geterr(pd));
    return false;
  }

  recv += stat.ps_recv;
  drops += stat.ps_drop;

  return true;
}

void PcapHandlerComponent::printStats(void)
{
  if (recv > 0)
    pandora_info("[pcap] recv:" << recv << setw(10) << " drops:" << drops);
}
