/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/fcntl.h>
#include <libpandora/conf/unistd.h>
#include <sys/socket.h>
}

#include "sendercomponent.h"

#include <pandora_components/requestpacket.h>
#include <libpandora/select.h>
#include <libpandora/common_options.h>
#include <libpandora/netutil.h>

component_export(SenderComponent, RequestPacket,);

static bool doRead(int fd, void *data);
static void doFinish(int fd, void *data);

#define REQS_DEBUG 1

#if REQS_DEBUG
static int reqsent = 0, reqdone = 0, reqerr = 0;
#endif

SenderComponent::SenderComponent(void) 
  : sel(new Select(0)), ctimeout(0), keepalive(false), ipbind(0)
{
  registerOption("ctimeout", &ctimeout);
  registerOption("keepalive", &keepalive);
  registerComplexOption(inet_addr_option, "bind", &ipbind);
}

SenderComponent::~SenderComponent(void)
{
  __DELETE(sel);
}

bool SenderComponent::add(Packet *pkt)
{
  RequestPacket *rpp = static_cast<RequestPacket *>(pkt);
  pandora_assert(rpp->port != 0 && rpp->addr != 0);

#if REQS_DEBUG
  pandora_debug("[requests: " << reqsent << " " << reqdone << " "
	       << reqerr << "]");
#endif
  int fd = openclient(rpp->port, rpp->addr, rpp->tcp, ctimeout, ipbind);

  if (fd < 0) {
    pandora_warning("failed sending request");
#if REQS_DEBUG
    ++reqerr;
#endif
    goto finished;
  }

  if (net_writeblock(fd, rpp->req.getData(), rpp->req.getLength()) < 0) {
    pandora_pwarning("net_writeblock");
    close(fd);
#if REQS_DEBUG
    ++reqerr;
#endif
    goto finished;
  }

#if REQS_DEBUG  
  ++reqsent;
  //pandora_debug("[adding  file desc: #" << fd << "]");
#endif

#if 1
  rpp->refer();
  sel->registerFd(fd, (void *)rpp, &doRead, &doFinish);
#else
  sel->registerFd(fd, (void *)this, &doRead, &doFinish);
#endif

 finished:
  cleanPacket(rpp);
  while (sel->select() > 0) /* empty */;

  return false;
}   

void SenderComponent::cleanup(void) 
{
  if (sel != NULL) {
    while (sel->select() > 0) /* empty */;
    sel->cleanup();
#if REQS_DEBUG
  pandora_debug("[requests: " << reqsent << " " << reqdone << " "
	       << reqerr << "]");
#endif
  }
}

static bool doRead(int fd, void *)
{
  static char buf[BUFSIZ];
  int n = recv(fd, buf, sizeof(buf), 0);
#if REQS_DEBUG > 1
  pandora_debug("[reading file desc: #" << fd << " (" << n << ")]");
#endif
  switch(n) {
  case 0: 
    return true;
  case -1: 
    if (errno == EINTR) return false; 
    pandora_pwarning("read");
#if REQS_DEBUG
    ++reqerr;
    --reqdone;
#endif
    return true;
  default:
    break;
  }
  return false;
}

static void doFinish(int fd, void *pkt)
{
#if REQS_DEBUG > 1
  pandora_debug("[closing file desc: #" << fd << "]");
#endif
#if REQS_DEBUG
  ++reqdone;
#endif

#if 1
  RequestPacket *rpp = reinterpret_cast<RequestPacket *>(pkt);
  rpp->release();
  cleanPacket(rpp);
#endif

  close(fd);
}
