/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>

packet_export(TCPPacket, IPPacket);

TCPPacket::TCPPacket(IPPacket* ipp) 
  : length(0), flags(0), cksum(0), gap(false), eof(false), hint(false)
{
  if (ipp == NULL) return;

  u_int datalen = ipp->dlength();

  if (datalen < sizeof(struct tcphdr)) {
    if (datalen > 0) {
      pandora_warning("truncated TCP packet (headers)");
      cleanPacket(ipp);
    } else { // let suppose this has been made on purpose...
      timeStamp = 	ipp->timeStamp;  
      packetSetUp(ipp);
    }
      return;
  }

  struct tcphdr *tp = (struct tcphdr *)(ipp->data());
  u_int hlen = TH_OFF(tp) * 4;

  if (datalen < hlen) {
    pandora_notice("truncated TCP packet (options)");
#if 0
    cleanPacket(ipp);
    return;
#else
    hlen = datalen;
#endif
  }
  
  length = 	ipp->length - hlen;
  sport = 	tp->th_sport;
  dport = 	tp->th_dport;
  flags = 	tp->th_flags;
  seq = 	ntohl(tp->th_seq);
  ack = 	ntohl(tp->th_ack);
  cksum = 	tp->th_sum;
  (ipp->_data).move(hlen);

  timeStamp = 	ipp->timeStamp;  
  packetSetUp(ipp);
}

TCPPacket::TCPPacket(const TCPPacket& x) 
  : Packet(x), sport(x.sport), dport(x.dport), 
    length(0), flags(x.flags),  seq(x.seq), ack(x.ack), 
    cksum(x.cksum), gap(x.gap), eof(x.eof), 
    hint(x.hint)
{
  /* empty */
}

TCPPacket& TCPPacket::operator= (const TCPPacket& x) 
{
  Packet::operator=(x);
  sport = x.sport; dport = x.dport; 
  length = 0; flags = x.flags; seq = x.seq; ack = x.ack; 
  cksum = x.cksum; gap = x.gap; eof = x.eof; 
  hint=x.hint;
  return *this;
}

void TCPPacket::print(ostream *f)
{
  locatePacket(IPPacket, ipp, this);
  if (ipp == NULL) return;

  *f << timeStamp << '\t'
     << "[tcp] "
     << ipp->src << ':' << ntohs(sport) << ' ' 
     << ipp->dst << ':' << ntohs(dport) << ' ';
  
  *f << length << '/' << ipp->dlength() << ' ' << seq << ' ';
  if (flags & TH_SYN) *f << "SYN" << ' ';
  if (flags & TH_FIN) *f << "FIN" << ' ';
  if (flags & TH_RST) *f << "RST" << ' ';
  if (flags & TH_ACK) *f << "ACK" << ' ' << ack << ' ';
  *f << (gap ? "* " : "");
  *f << endl;
}

size_t TCPPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(sport);
  serialVar(dport);

  beginLevel(0);
  serialVar(length);
  serialVar(flags);
  serialVar(seq);
  serialVar(ack);
  serialVar(cksum);
  serialVar(gap);
  serialVar(eof);
  
  return count;
} 

size_t TCPPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(sport);
  unserialVar(dport);

  beginLevel(0);
  unserialVar(length);
  unserialVar(flags);
  unserialVar(seq);
  unserialVar(ack);
  unserialVar(cksum);
  unserialVar(gap);
  unserialVar(eof);

  return count;
}


extern_pandora(algo, bool, tcpcnx, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(TCPPacket, tcpp, pkt);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL) return false;

  if ((tcpp->sport < tcpp->dport) 
      || ((tcpp->sport == tcpp->dport) 
	  && (ipp->src.s_addr < ipp->dst.s_addr))) {
    k->set(tcpp->sport, ipp->dst.s_addr, tcpp->dport, ipp->src.s_addr);
  } else {
    k->set(tcpp->dport, ipp->src.s_addr, tcpp->sport, ipp->dst.s_addr);
  }
  return true;
}

extern_pandora(algo, bool, tcpquad, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(TCPPacket, tcpp, pkt);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL) return false;

  k->set(ipp->dst.s_addr, ipp->src.s_addr, tcpp->dport, tcpp->sport);
 
  return true;
}

#include <libpandora/quadkey.h>

extern_pandora(algo, DmxKey *, tcpcnx2, (Packet *pkt))
{
  locatePacket0(TCPPacket, tcpp, pkt);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL) return NULL;

  if ((tcpp->sport < tcpp->dport) 
      || ((tcpp->sport == tcpp->dport) 
	  && (ipp->src.s_addr < ipp->dst.s_addr))) {
    return 
      new QuadKey(tcpp->sport, ipp->dst.s_addr, tcpp->dport, ipp->src.s_addr);
  } else {
    return 
      new QuadKey(tcpp->dport, ipp->src.s_addr, tcpp->sport, ipp->dst.s_addr);
  }
}


