/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h> 

extern "C" {
#include <netinet/in.h>
#include <netinet/in_systm.h>
 	   }

#include <libpandora/pandora_alloc.h>
#include <list>

#include "tcpreordercomponent.h"
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>

component_export(TCPReorderComponent, TCPPacket, TCPPacket);

#define DIRECTION	0

bool TCPReorderComponent::dispatch(void)
{
  if (state < TCPS_ESTABLISHED && !force) return false;

  while (!plist.empty()) {
    bool eof = false;
    
    TCPPacket *tcpp = *plist.begin();
    pandora_assert(tcpp != NULL);

    if ((tcpp->flags & TH_RST) && (tcpp->seq == 0)) tcpp->seq = nextSeq;

    if (state >= TCPS_ESTABLISHED && (SEQ_LT(tcpp->seq, nextSeq))) {
     plist.pop_front();     
     cleanPacket(tcpp);
     continue;
    }
  
    if (force) {
      //pandora_debug("force");
      if(state < TCPS_ESTABLISHED) { // we have missed SYN packet
#if 1
        state = TCPS_ESTABLISHED;
#else
        flush();
        return false;
#endif
      }
      tcpp->gap = (tcpp->seq != nextSeq);
      nextSeq = SEQ_ADD(tcpp->seq, tcpp->length);
      force = false;
    } else {
      if (nextSeq != tcpp->seq) return false;
      nextSeq = SEQ_ADD(tcpp->seq, tcpp->length);
    }
    
    plist.pop_front();
    bytesStored -= tcpp->length;
    
    if (force && plist.empty()) tcpp->eof = true;
    else eof = tcpp->eof = tcpp->flags & (TH_FIN | TH_RST);
    
    
    //pandora_debug(plist.size() << " (" << disp_count << ", " << bytesStored << ") "
    //     << tcpp);
    
    push(tcpp);

    if (eof) {
      flush();
      return true;
    }
  }
  return false;
}



bool TCPReorderComponent::add(Packet *obj)
{
  TCPPacket *tcpp = (TCPPacket *) obj;
  locatePacket(IPPacket, ipp, tcpp);
  pandora_assert(ipp != NULL);

  bool doDispatch = false, fin = false;;
  register PListIter prev,next;
  register PListIter tmp,tmp2;

#if 0
  if (TCPS_HAVERCVDFIN(state)) {
    //pandora_warning(tcpp << ": should not happen");
    cleanPacket(tcpp);
    return true;
  }
#endif
  
  pandora_assert(tcpp != NULL);
  //cerr << "> "; tcpp->print();

#if 1
  // fast processing path
  if (state >= TCPS_ESTABLISHED 
      && plist.empty()
      && tcpp->seq == nextSeq) {
    nextSeq = SEQ_ADD(tcpp->seq, tcpp->length);
    fin = tcpp->eof = tcpp->flags & (TH_FIN | TH_RST);
    push(tcpp);
    goto nodisp;
  }
#endif

  // slow processing path
  if (tcpp->flags & TH_SYN) {
    nextSeq = SEQ_ADD(tcpp->seq, 1);
    state = TCPS_ESTABLISHED;
    doDispatch = true;
    push(tcpp);

    goto nodisp;
  } 
  
  if (state >= TCPS_ESTABLISHED && (SEQ_LT(tcpp->seq, nextSeq))) {
    if (SEQ_LT(SEQ_ADD(tcpp->seq, tcpp->length), nextSeq)) {
      cleanPacket(tcpp);
      goto finished;
    } else {
      int32_t i=SEQ_SUB(nextSeq, tcpp->seq);
      tcpp->seq=SEQ_ADD(tcpp->seq, i);
      if (tcpp->length > i) {
	tcpp->length -= i;
	(ipp->_data).move(i);
      } else {
        cleanPacket(tcpp);
	goto finished;
      }
    }
  }
  
#if DIRECTION
  prev=plist.end();
  if (!plist.empty()) {
    for (next=plist.begin(); next!=plist.end(); ++next) {

      pandora_assert((*next) != NULL);

      if (SEQ_GEQ((*next)->seq, tcpp->seq)) break;
      prev=next;
    }
  } else next=plist.end();
#else
  prev=plist.end();
  if (!plist.empty()) {

    for (next=plist.end(); next!=plist.begin(); --next) {
      tmp = next;
      --tmp;

      pandora_assert((*tmp) != NULL);
      if (SEQ_LT((*tmp)->seq, tcpp->seq)) break;
    }
    if (next != plist.begin()) prev = tmp;

  } else next=plist.end();
#endif
  

  if ((prev != plist.end()) && 
      (SEQ_LT(tcpp->seq, SEQ_ADD((*prev)->seq, (*prev)->length)))) {
    
    int32_t i=SEQ_SUB(SEQ_ADD((*prev)->seq,(*prev)->length),tcpp->seq);

    if (tcpp->length > i) {
      tcpp->seq = SEQ_ADD(tcpp->seq, i);
      tcpp->length -= i;
      (ipp->_data).move(i);
    } else {
      cleanPacket(tcpp);
      goto finished;
    }
  }
    
  for (tmp = next; tmp != plist.end(); tmp = tmp2) {
    TCPPacket *tcpp2 = *tmp;
    tmp2 = tmp;
    ++tmp2;
    if (SEQ_GEQ(tcpp2->seq, SEQ_ADD(tcpp->seq, tcpp->length))) break;
    
    int32_t i=SEQ_SUB(SEQ_ADD(tcpp->seq,tcpp->length),(*next)->seq);

    bytesStored -= pandora_min(tcpp2->length, i);

    if (tcpp2->length > i) {
      locatePacket(IPPacket, ipp2, tcpp2);
      pandora_assert(ipp2 != NULL);

      tcpp2->seq = SEQ_ADD((*tmp)->seq, i);
      (ipp2->_data).move(i);
      tcpp2->length -= i;
    } else {
      next = tmp2;
      plist.erase(tmp);
      cleanPacket(tcpp2);
    }
  }

  doDispatch = (prev == plist.end());
  
  if (true /* discard strange sequenced packets */) {
    u_int32_t prevSeq = (doDispatch ? nextSeq : (*prev)->seq);
    if ((state >= TCPS_ESTABLISHED)
        && (SEQ_GT(tcpp->seq, SEQ_ADD(prevSeq, bytesMax)))
        && !(tcpp->flags & TH_RST)) {
      cleanPacket(tcpp);
      goto finished;
    }
  }
  
  (void) plist.insert(next, tcpp);  
  
  bytesStored += tcpp->length;
  
  if (plist.size() > (u_int)packetsMax) {
#if 1
    force = true;
#else
    state = TCPS_ESTABLISHED;
#endif
    doDispatch = true;
  }

 finished:  
  if (doDispatch) fin = dispatch();

 nodisp:  
  return fin;
}

void TCPReorderComponent::cleanup(void) 
{
  force = true;
  (void)dispatch();
  flush();
  nextSeq = 0; bytesStored = 0; state = TCPS_LISTEN; eof = false;
  force = false;
}

void TCPReorderComponent::flush(void) 
{
  if (!plist.empty()) {
    for (PListIter ptr = plist.begin(); ptr != plist.end(); ++ptr) {
      TCPPacket *tcpp = *ptr;
      cleanPacket(tcpp);
    }
    plist.clear();
  }
}

