/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
#include "stdlib.h"
	   }

#include <pandora_components/w2npacket.h>
#include <pandora_components/fseventpacket.h>
#include <pandora_components/syscallpacket.h>
#include <pandora_components/valuepacket.h>
#include <libpandora/timeval.h>
#include <libpandora/netutil.h>
#include <libpandora/serialize.h>

packet_export(W2NPacket, TextValuePacket);

W2NPacket::W2NPacket(int)
  : opcode(-1), version(0), addr(0), port(0), asnum(0)
{
  setTimeStamp();
}

W2NPacket::W2NPacket(TextValuePacket *vp)
  : opcode(-1), version(0), addr(0), port(0), asnum(0)
{
  if (vp == NULL) return;  

  char *line = (vp->val).data();
  const char *delims = " \t\r\n";

  char *u = strtok(line, delims);
  if (u == NULL) {
    //pandora_warning("NULL url");
    cleanPacket(vp);
    return;
  } else {
    url.init(u);
  }  

  setOpCode(SUBSCRIBE);

  timeStamp = vp->timeStamp;
  cleanPacket(vp);
}

W2NPacket::W2NPacket(const W2NPacket& x) 
  : Packet(x), 
    opcode(x.opcode), version(x.version), 
    addr(x.addr), port(x.port), asnum(x.asnum),
    url(x.url), url2(x.url2), server(x.server), 
    path(x.path), path2(x.path2)
{ 
}

W2NPacket& W2NPacket::operator= (const W2NPacket& x)
{
  Packet::operator=(x);
  opcode = x.opcode; version = x.version;
  addr = x.addr; port = x.port; asnum = x.asnum;
  url = x.url; url2 = x.url2; server = x.server;
  path = x.path; path2 = x.path2; 
  return *this;
}

void W2NPacket::setOpCode(const FSEventPacket *fsep)
{
  if (fsep == NULL) return;

  switch (fsep->type) {
  case FSEventPacket::CREATED: 	case FSEventPacket::MODIFIED: 	
    				opcode = UPDATE; 	break;
  case FSEventPacket::DELETED: 	opcode = UNLINK;  	break;
  default: 			opcode = ERROR;		break;
  }
}

void W2NPacket::print(ostream *f) 
{
  *f << timeStamp << '\t' 
     << "[w2n] " << opcode << " " << url << " ";
  *f  << endl;
}

W2NPacket *W2NPacket::recv_raw(int fd, char *temp, size_t maxlen)
{
  if (fd < 0) return NULL;
  int nbytes = net_readblock(fd, temp, maxlen);
  if (nbytes <= 0) return NULL;

  W2NPacket *w2np = new W2NPacket(0);
  if ((int)w2np->read(temp, 0) != nbytes) {
    pandora_warning("number of bytes decoded mismatch");
    cleanPacket(w2np);
      return NULL;
  }

  if (w2np->getOpCode() == W2NPacket::ERROR) {
    pandora_warning("discarding invalid packet");
    cleanPacket(w2np);
    return NULL;
  }

  return w2np;
}

size_t W2NPacket::write(char *str, size_t maxlen, int)
{
  size_t count = 0;

  serialVar(opcode);
  switch(getOpCode()) {
  case UPDATE: case UNLINK: case MKDIR: case RMDIR: 
    serialVar(version);
    serialVar(url);
    break;
  case RENAME:
    serialVar(version);
    serialVar(url);
    serialVar(url2);
    break;
 
  case MONITOR_REGISTER: 
    serialVar(asnum);
    serialVar(server);
    serialVar(port);
    break;

  case SUBSCRIBE: 
    serialVar(url);
    serialVar(port);
    break;

  case SERVCE_PORT: case UNSUBSCRIBE: case LIST:
  case CLIENT_BIND: case NS_REGISTER: case NS_BIND:
  case NS_UNREGISTER: case MONITOR_UNREGISTER: 
    break;

  default:
    break;
  }

  return count;
}

size_t W2NPacket::read(const char *str, int)
{
  size_t count = 0;

  unserialVar(opcode);
  switch(getOpCode()) {
  case UPDATE: case UNLINK: case MKDIR: case RMDIR: 
    unserialVar(version);
    unserialVar(url);
    break;
  case RENAME:
    unserialVar(version);
    unserialVar(url);
    unserialVar(url2);
    break;
 
  case MONITOR_REGISTER: 
    unserialVar(asnum);
    unserialVar(server);
    unserialVar(port);
    break;

  case SUBSCRIBE: 
    unserialVar(url);
    unserialVar(port);
    break;

  case SERVCE_PORT: case UNSUBSCRIBE: case LIST:
  case CLIENT_BIND: case NS_REGISTER: case NS_BIND:
  case NS_UNREGISTER: case MONITOR_UNREGISTER: 
    break;

  default:
    break;
  }

  return count;
}

