/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <libpandora/conf/fcntl.h>
#include <libpandora/conf/poll.h>
#include <libpandora/conf/wait.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/usleep.h>
extern int flock(int, int);
#ifndef LOCK_SH
#define LOCK_SH 1
#endif
#ifndef LOCK_EX
#define LOCK_EX 2
#endif	     
#ifndef LOCK_NB
#define LOCK_NB 4
#endif	     
#ifndef LOCK_UN
#define LOCK_UN 8
#endif
	   }

#include <libpandora/error.h>
#include <libpandora/netutil.h>
#include <libpandora/setsignal.h>

const static char lockfile[] = 	"/tmp/pandora.lock";
const static int  max_errors = 	20;
const static int  mgr_port = 	0 /* 23123 */;
const static int  sleep_ms = 	1000;
const static int  do_fork =     0;
static char      **cmd  = 	NULL;

static char      *prog_name =	NULL;
static pid_t      lockpid = 	0;
static pid_t      dead =	0;
static int        lockfd = 	-1;
static int	  status =	0;
static int        running =	0;


extern "C" RETSIGTYPE child_wait(int signum)
{
  int flags = WUNTRACED;
  if (signum != 0) {
    if (signum != SIGCHLD) pandora_error("caugth signal: " << signum);
    flags |= WNOHANG;
  }
  status = 0;
  dead = waitpid(-1, &status, flags);
  if (dead < 0)  pandora_pwarning("waitpid");
  if (dead == 0) pandora_warning("waitpid: chlid resurected!");
  //pandora_debug("SIGCHLD: #" << dead);
  running = 0;
  return RETSIGVAL;
}


static void run_cmd(void)
{
  (void) setsignal(SIGCHLD, &child_wait);
  running = 1;

  pid_t spid = fork();
  if (spid < 0) pandora_perror("fork");
    
  if (spid == 0) {
    execv(*cmd, cmd);
    pandora_perror(*cmd);
  }

  pandora_debug("run child with pid #" << spid);

  if (lseek(lockfd, sizeof(pid_t), SEEK_SET) != sizeof(pid_t))
    pandora_perror("lseek");

  if (write(lockfd, (char *)&spid, sizeof(pid_t)) != sizeof(pid_t))
    pandora_perror("write");
}


void check_cmd(void)
{
  struct stat stat_buf;

  if (cmd == NULL || *cmd == NULL) pandora_error("invalid null command");
  if (lstat(*cmd, &stat_buf) < 0) pandora_perror(*cmd);
  if (!(S_ISREG(stat_buf.st_mode))
      || ((geteuid() == stat_buf.st_uid) && !(stat_buf.st_mode & S_IXUSR))
      || ((getegid() == stat_buf.st_gid) && !(stat_buf.st_mode & S_IXGRP))
      || !(stat_buf.st_mode & S_IXOTH)) 
    pandora_error(*cmd << ": Permission denied");
  return;
}


static int locker(int repl)
{
  if (lockpid != 0) {
    pandora_debug("previous lock by process #" << lockpid << " dropped");
  }

  check_cmd();

  if ((do_fork) && (fork() != 0)) return 0;

  pid_t pid = getpid();
  pandora_debug("locker process #" << pid);

  dead = 0;

  if (lseek(lockfd, 0, SEEK_SET) != 0)
    pandora_perror("lseek");

  if (write(lockfd, (char *)&pid, sizeof(pid_t)) != sizeof(pid_t))
    pandora_perror("write");


  if (repl) {
    pid_t spid = 0;
    
    pandora_debug("replacing dead locking process #" << lockpid);

    if (lseek(lockfd, sizeof(pid_t), SEEK_SET) != sizeof(pid_t))
      pandora_perror("lseek");
    
    if (read(lockfd, (char *)&spid, sizeof(pid_t)) < 0)
      pandora_perror("read");

    pandora_debug("former child had pid #" << spid);

    if (::flock(lockfd, LOCK_EX) < 0) pandora_perror("flock");
    pandora_debug("control taken");
  }

  int error_count = 0;

 retry:
  int listenfd = 0;
  char buf[BUFSIZ];
  int len = 0;
  struct sockaddr_in from;
  socklen_t fromlen;

  if (mgr_port > 0) {
    listenfd = openserver(mgr_port, false);
    if (listenfd < 0) pandora_error("cannot open server on port " << mgr_port);
    pandora_debug("listening on port udp/" << mgr_port);
    
    
    /* Sometimes it is not possible to get the socket name at first,
       so try to get here to make it ready for the next time */
    memset((char *)&from, 0, sizeof(struct sockaddr_in));
    if (getsockname(listenfd, (struct sockaddr *)&from, &fromlen) < 0)
      pandora_perror("getsockname");
  } else {
    if (!running) run_cmd();
  }

  do {
    struct pollfd ufd;
    memset((char *)&ufd, 0, sizeof(ufd));
    ufd.fd = listenfd;
    ufd.events = POLLIN;

    int p = 0;

    if (listenfd > 0) {
      if (!dead) p = poll(&ufd, 1, -1);
      
      if ((p < 0) && (errno != EINTR)) {
	pandora_pwarning("poll");
	continue;
      }
    } else {
      pause();
    }

    if (dead) {
      if (WIFEXITED(status)) {
	pandora_debug("child #" << dead << " terminated normally");
	error_count = 0;
	if (listenfd == 0) break;
      } else {
	pandora_debug("child #" << dead << " died, restarting it");
	run_cmd();
      }
      dead = 0;
      continue;
    }

    if (p == 0) {
      pandora_warning("poll: timeout expired");
      continue;
    }

    if (listenfd > 0) {
      if (ufd.revents != POLLIN) {
	pandora_warning("connection closed");
	close(listenfd);
	goto retry;
      }  
      
      memset((char *)&from, 0, sizeof(struct sockaddr_in));
      if ((len = recvfrom(listenfd, buf, sizeof(buf), 0, 
			  (struct sockaddr *)&from, &fromlen)) < 0) {
	pandora_pwarning("recvfrom");
	continue;
      }

      if (from.sin_addr.s_addr == 0) {
	pandora_warning("cannot get socket name, skipping");
	continue;
      }
    }

    if (!running) {
      run_cmd();
      usleep(1000*sleep_ms);
    }

    if (listenfd > 0) {
      len = 0;
      if (sendto(listenfd, buf, len, 0, 
		 (struct sockaddr *)&from, fromlen) < len) {
	pandora_pwarning("sendto");
	continue;
      }
    }

    error_count = 0;
  } while (error_count++ < max_errors);

  if (error_count >= max_errors) {
    if (running) (void) child_wait(0);
    pandora_warning("too many errors encountered, exiting");
  }

  unlink(lockfile);
  return 1;
}


static int locked(void)
{
  int ret = kill(lockpid, 0);
  if ((ret < 0) && (errno != ESRCH))
    pandora_perror("kill");

  if (ret == 0) {
    pandora_debug("already locked by process #" << lockpid);
    return 1;
  } else {
    return locker(1);
  }
}


int main(int argc, char *argv[])
{
  set_verbosity(10);
  prog_name = argv[0];
  cmd = argv+1;

  lockfd = open(lockfile, 
	    O_RDWR|O_CREAT, 
	    S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);

  if (lockfd < 0)
    pandora_perror(lockfile);

  int ret = ::flock(lockfd, LOCK_EX|LOCK_NB);

  if ((ret < 0) && (errno != EWOULDBLOCK))
    pandora_perror("flock");

  if (read(lockfd, (char *)&lockpid, sizeof(pid_t)) < 0) 
    pandora_perror("read");

  if ((ret < 0) ? locked() : locker(0))
    close(lockfd);

  return 0;
}
