/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "arc.h"

using namespace Papyrus;

Arc::Arc(double angle_start, double angle_finish, double radius) :
        Shape(),
m_radius(radius) {
  if (angle_start < angle_finish) {
    m_angle_start = angle_start;
    m_angle_finish = angle_finish;
  } else {
    m_angle_start = angle_finish;
    m_angle_finish = angle_start;
  }
}

Arc::~Arc() {}

double Arc::get_radius() {
    return m_radius;
}

void Arc::set_radius(double r) {
    m_radius = r;
    on_changed(RENDER_RADIUS);
}

void Arc::draw_shape( cairo_t * cairo ) {
    cairo_arc(cairo, 0.0, 0.0, m_radius, m_angle_start, m_angle_finish);
    if (m_angle_finish - m_angle_start < 2.0*M_PI) {
        cairo_line_to(cairo, m_x, m_y);
        cairo_line_to(cairo, m_x+m_radius, m_y);
    }
}

double Arc::get_start_angle( ) {
    return m_angle_start;
}

double Arc::get_finish_angle( ) {
    return m_angle_finish;
}

void Arc::set_angles( double start, double finish ) {
    if (start < finish) {
        m_angle_start = start;
        m_angle_finish = finish;
    } else {
        m_angle_start = finish;
        m_angle_finish = start;
    }

    on_changed(RENDER_ANGLE_START|RENDER_ANGLE_FINISH);
}

void Arc::on_changed(int i) {
  if (i & (RENDER_RADIUS|RENDER_ANGLE_START|RENDER_ANGLE_FINISH)) {
    BBox tmpbbox;

    tmpbbox.set_xywh(0.0, 0.0, 0.0, 0.0);

    tmpbbox.update(cos(m_angle_start)*m_radius, sin(m_angle_start)*m_radius);
    tmpbbox.update(cos(m_angle_finish)*m_radius, sin(m_angle_finish)*m_radius);
    if (m_angle_start <= M_PI/2.0 && m_angle_finish >= M_PI/2.0)
        tmpbbox.update(0.0, m_radius);
    if (m_angle_start <= M_PI && m_angle_finish >= M_PI)
        tmpbbox.update(-m_radius, 0.0);
    if (m_angle_start <= 3.0*M_PI/2.0 && m_angle_finish >= 3.0*M_PI/2.0)
        tmpbbox.update(0.0, -m_radius);
    if (m_angle_start <= 2.0*M_PI && m_angle_finish >= 2.0*M_PI)
        tmpbbox.update(m_radius, 0.0);

    m_bbox = tmpbbox;
    Drawable::on_changed(i|RENDER_BBOX);
  }

  Drawable::on_changed(i);
}

void Arc::on_changed_proxy( int i )
{
  on_changed(i);
}
