// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "db.h"
#include "in_impl.h"

#include <Poco/SAX/SAXParser.h>
#include <Poco/SAX/SAXException.h>
#include <Poco/SAX/ErrorHandler.h>
#include <Poco/SAX/ContentHandler.h>
#include <Poco/SAX/Attributes.h>
#include <Poco/SAX/InputSource.h>

using namespace Poco;
using namespace Poco::XML;

//=== XMLFile ==============================================================
XMLFile::XMLFile (const string& option_key)
	: FileInputImpl(option_key, "xml")
{
}

	// error_handler
	class error_handler: public ErrorHandler {
		public:
		error_handler ();
		virtual void fatalError (const SAXException& exc);
		virtual void error (const SAXException& exc);
		virtual void warning (const SAXException& exc);
		protected:
		void handle (const string& s, const SAXException& exc);
	};

	error_handler::error_handler ()
	{
	}

	void error_handler::fatalError (const SAXException& exc)
	{
		handle("XML fatal", exc);
	}

	void error_handler::error (const SAXException& exc)
	{
		handle("XML error", exc);
	}

	void error_handler::warning (const SAXException& exc)
	{
		handle("XML warning", exc);
	}

	void error_handler::handle (const string& s, const SAXException& exc)
	{
		string msg(exc.displayText());
		string::size_type pos = msg.find("Exception: ");
		if (pos != string::npos)
			msg.erase(0, pos + 11);
		throw Xception(s + ": " + msg);
	}

	// content_handler
	class content_handler: public ContentHandler {
		enum Format {unspecified, pdr};
		SAXParser& m_parser;
		Database::CollectionElements& m_elements;
		Database& m_database;
		Format m_format;
		string m_current_collection;
		char m_current_collection_type;
		public:
		content_handler (SAXParser& parser, Database::CollectionElements& elements, Database& database);
		virtual void characters (const XMLChar ch[], int start, int length) {}
		virtual void startDocument () {}
		virtual void endDocument () {}
		virtual void startElement (const XMLString& uri, const XMLString& localName, const XMLString& qname, const Attributes& attrList);
		virtual void endElement (const XMLString& uri, const XMLString& localName, const XMLString& qname);
		virtual void startPrefixMapping (const XMLString& prefix, const XMLString& uri) {}
		virtual void endPrefixMapping (const XMLString& prefix) {}
		virtual void ignorableWhitespace (const XMLChar ch[], int start, int length) {}
		virtual void processingInstruction (const XMLString& target, const XMLString& data) {}
		virtual void setDocumentLocator (const Locator* loc) {}
		virtual void skippedEntity (const XMLString& name) {}
	};

	content_handler::content_handler (SAXParser& parser, Database::CollectionElements& elements, Database& database)
		: m_parser(parser)
		, m_elements(elements)
		, m_database(database)
		, m_format(unspecified)
		, m_current_collection()
		, m_current_collection_type('\0')
	{
	}

	// the idea is to parse several XMLformats using one parser, so we
	// have to distinguish these formats, we use the name of the first
	// element for that, this element sets the m_format variable
	void content_handler::startElement (const XMLString& uri, const XMLString& localName, const XMLString& qname, const Attributes& attrList)
	{
		switch (m_format)
		{
			case unspecified:
			{
				if (localName == "pdr")
				{
					m_format = pdr;
					return;
				}

				// other formats ...

				break;
			}

			case pdr:
			{
				if (localName == "collection")
				{
					m_current_collection = attrList.getValue(uri, "name");
					m_current_collection_type = m_database.GetCollectionType(m_current_collection);
					return;
				}

				if (localName == "item")
				{
					ptime timestamp(not_a_date_time);
					{
						string s(attrList.getValue(uri, "datetime"));
						trim(s);
						try
						{
							if (s.find('T') != string::npos)
								timestamp = from_iso_string(s);
							else
								timestamp = lexical_cast<ptime>(s);
						}
						catch (bad_lexical_cast)
						{
							throw Xception(format("value error, %s is not a datetime") % s);
						}
					}

					any a;
					{
						string s(attrList.getValue(uri, "value"));
						switch (m_current_collection_type)
						{
							case 'n':
							{
								try
								{
									a = lexical_cast<double>(s);
								}
								catch (bad_lexical_cast)
								{
									throw Xception(format("value error, %s is not a double") % s);
								}
								break;
							}
							case 'r':
							{
								try
								{
									a = lexical_cast<Ratio>(s);
								}
								catch (bad_lexical_cast)
								{
									throw Xception(format("value error, %s is not a Ratio") % s);
								}
								break;
							}
							default:
							{
								a = s;
								break;
							}
						}
					}

					m_elements.push_back(Database::CollectionElement(m_current_collection, timestamp, a));

					return;
				}

				break;
			}

			// other formats ...

			default:
				break;
		}
	}

	void content_handler::endElement (const XMLString& uri, const XMLString& localName, const XMLString& qname)
	{
		switch (m_format)
		{
			case pdr:
			{
				if (localName == "pdr")
				{
					m_format = unspecified;
					return;
				}

				if (localName == "collection")
				{
					m_current_collection.clear();
					m_current_collection_type = '\0';
					return;
				}

				break;
			}

			// other formats ...

			default:
				break;
		}
	}

void XMLFile::ProcessFile (const Config& config, Database& database, ifstream& ifs, Database::CollectionElements& elements) const throw (Xception)
{
	string msgs;
	EncodingNames names;
	GetEncodingNames(names);

	// parse XML
	try
	{
		SAXParser parser;

		error_handler eh;
		parser.setErrorHandler(&eh);

		foreach (const string& name, names)
		{
			parser.addEncoding(name, &GetEncoding(name));
		}

		content_handler ch(parser, elements, database);
		parser.setContentHandler(&ch);

		InputSource isrc(ifs);
		parser.parse(&isrc);
	}
	catch (const Xception& )
	{
		throw;
	}
	catch (...)
	{
		throw Xception("xml parse error");
	}
}
