// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

//=== PocoDatabaseImpl (abstract base class) ===============================
PocoDatabaseImpl::PocoDatabaseImpl (const string& connect, bool verbose, const string& KEY, Connector* pConnector)
	: PocoDatabase(connect, verbose, KEY, pConnector)
{
}

PocoDatabaseImpl::~PocoDatabaseImpl ()
{
}

void PocoDatabaseImpl::Connect () throw (Xception)
{
	return PocoDatabase::Connect();
}

string PocoDatabaseImpl::GetNewTblname () const
{
	set<string> tblnames;
	foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
	{
		tblnames.insert(cmi.m_tblname);
	}
	set<string>::const_iterator I = tblnames.end();
	string s(*(--I));
	s.erase(0, 1); // C
	int i = lexical_cast<int>(s);
	return (format("C%d") % ++i).str();
}

char PocoDatabaseImpl::GetCollectionTypeFromValue (const any& a) const
{
	if (a.type() == typeid(double))
		return 'n';

	if (a.type() == typeid(Ratio))
		return 'r';

	if (a.type() == typeid(string))
		return 't';

	throw Xception("invalid type");
}

void PocoDatabaseImpl::ListCollections () throw (Xception)
{
	if (m_verbose)
		cout << "listing collections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {6, 8, 6, 7, 20}; // real column widths
			size_t width = widths[idx];
			string result(m_s);
			while (result.length() < width)
				result += ' ';
			if (result.length() > width)
				result.erase(width - 1);
			return result + ' '; // + 1 space as column delimiter
		}
	};

	try
	{
		cout << "  " << fill("name")(0) << fill("type")(1) << fill("table")(2) << fill("recs")(3) << fill("first")(4) << "last" << endl;

		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			cout << "  " << fill(cmi.m_collection)(0);

			switch (cmi.m_type)
			{
				case 'n':	cout << fill("numeric")(1); break;
				case 'r':	cout << fill("ratio")(1); break;
				case 't':	cout << fill("text")(1); break;
				default:	break;
			}

			cout << fill(cmi.m_tblname)(2);

			int count;
			string min, max;
			(*m_pSession) << "select count(*),min(t),max(t) from " << cmi.m_tblname << ";", into(count), into(min), into(max), now;
			cout << fill(lexical_cast<string>(count))(3) << fill(min)(4) << max << endl;
		}
	}
	catch (...)
	{
	}
}

void PocoDatabaseImpl::AddCollection (const string& name) throw (Xception) // "name[, n|r|t]"
{
	if (m_verbose)
		cout << "adding collection" << endl;

	try
	{
		// split collection name and type
		string n, t;
		{
			regex rx("\\s*([^,\\s]+)(?:\\s*,\\s*([nrt])\\s*)?");
			smatch mr;
			if (!regex_match(name, mr, rx))
				throw Xception(format("illegal collection name: %s") % name);
			n = mr[1];
			t = mr[2];
			if (t.empty())
				t = "n"; // numeric
		}

		// check for existing collection
		try
		{
			GetCollectionMetaInfo(n);
			throw Xception(format("collection already exists: %s") % n);
		}
		catch (...)
		{
			// ok, table doesn't exist
		}

		// build new table name
		const string& tblname = GetNewTblname();

		// do the database specific SQL job
		CreateCollectionInSchema(n, t[0], tblname);

		// create a CollectionMetaInfo
		CollectionMetaInfo cmi = {n, t[0], tblname};
		m_collectionMetaInfos.insert(cmi);
	}
	catch (...)
	{
		throw Xception(format("could not create collection: %s") % name);
	}
}

void PocoDatabaseImpl::DeleteCollection (const string& name) throw (Xception)
{
	if (m_verbose)
		cout << "deleting collection" << endl;

	try
	{
		// check for built-in collection
		if (name == "*" || name == "#")
			throw Xception(format("cannot delete built-in collection: %s") % name);

		// check for unknown collection
		const CollectionMetaInfo& cmi = GetCollectionMetaInfo(name);

		// do the database specific SQL job
		DropCollectionFromSchema(cmi.m_tblname);

		// forget the CollectionMetaInfo
		m_collectionMetaInfos.erase(cmi);
	}
	catch (...)
	{
		throw Xception(format("could not delete collection: %s") % name);
	}
}

void PocoDatabaseImpl::DeleteAllCollections () throw (Xception)
{
	if (m_verbose)
		cout << "deleting all collections" << endl;

	try
	{
		DBTransactor transactor(m_pSession);

		CollectionMetaInfos to_delete;
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			if (cmi.m_collection != "*" && cmi.m_collection != "#")
				to_delete.insert(cmi);
		}
		foreach (const CollectionMetaInfo& cmi, to_delete)
		{
			(*m_pSession) << "drop table " << cmi.m_tblname << ";", now;
			m_collectionMetaInfos.erase(cmi);
		}

		(*m_pSession) << "delete from TCollections where name not in ('*','#');", now;
		(*m_pSession) << "delete from C0;", now;
		(*m_pSession) << "delete from C1;", now;

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not delete collections");
	}
}

void PocoDatabaseImpl::ListRejections () throw (Xception)
{
	if (m_verbose)
		cout << "listing rejections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {20}; // real column widths
			size_t width = widths[idx];
			string result(m_s);
			while (result.length() < width)
				result += ' ';
			if (result.length() > width)
				result.erase(width - 1);
			return result + ' '; // + 1 space as column delimiter
		}
	};

	try
	{
		Statement select(*m_pSession);
		select << "select t,expr from TRejected order by t;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			cout << "  " << fill("timestamp")(0) << "expression" << endl;
			do {
				cout << "  " << fill(rs.value(0).convert<string>())(0) << rs.value(1).convert<string>() << endl;
			} while (rs.moveNext());
		}
	}
	catch (...)
	{
	}
}

void PocoDatabaseImpl::AddRejected (const ptime& timestamp, const string& expr) throw (Xception)
{
	try
	{
		const string& t = lexical_cast<string, ptime>(timestamp);
		(*m_pSession) << "insert into TRejected values (null,'" << t << "','" << expr << "';", now;
	}
	catch (...)
	{
		throw Xception("could not insert expression");
	}
}

void PocoDatabaseImpl::DeleteAllRejections () throw (Xception)
{
	if (m_verbose)
		cout << "deleting all rejections" << endl;

	try
	{
		(*m_pSession) << "delete from TRejected;", now;
	}
	catch (...)
	{
		throw Xception("could not delete rejections");
	}
}

char PocoDatabaseImpl::GetCollectionType (const string& name) const throw (Xception)
{
	return PocoDatabase::GetCollectionType(name);
}

void PocoDatabaseImpl::GetCollections (Collections& collections) const throw (Xception)
{
	foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
	{
		collections.insert(cmi.m_collection);
	}
}

void PocoDatabaseImpl::GetCollectionItems (const string& name, CollectionItems& items) const throw (Xception)
{
	try
	{
		const CollectionMetaInfo& cmi = GetCollectionMetaInfo(name);

		Statement select(*m_pSession);
		select << "select ";
		switch (cmi.m_type)
		{
			case 'r':	select << "t,n,d"; break;
			default:	select << "t,v"; break;
		}
		select << " from " << cmi.m_tblname << ";";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			do {
				ptime t = lexical_cast<ptime>(rs.value(0).convert<string>());
				any a;
				switch (cmi.m_type)
				{
					case 'n':	a = rs.value(1).convert<double>(); break;
					case 'r':	a = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
					case 't':	a = rs.value(1).convert<string>(); break;
				}
				items.insert(Database::CollectionItems::value_type(t, a));
			} while (rs.moveNext());
		}
	}
	catch (...)
	{
		throw Xception("could not get collection items");
	}
}

void PocoDatabaseImpl::AddCollectionsItems (const CollectionsItems& items) throw (Xception)
{
	try
	{
		DBTransactor transactor(m_pSession);

		foreach (const CollectionsItems::value_type& vt, items)
		{
			const CollectionMetaInfo* pCmi = NULL;
			try
			{
				 pCmi = &GetCollectionMetaInfo(vt.first);
			}
			catch (const Xception& )
			{
				// if the collection doesn't exist ...
				// ... create it now
				const string& n = vt.first;
				char t = GetCollectionTypeFromValue(vt.second.second);
				const string& tblname = GetNewTblname();

				CreateCollectionInSchema(n, t, tblname);

				CollectionMetaInfo cmi = {n, t, tblname};
				CollectionMetaInfos::iterator I = m_collectionMetaInfos.insert(cmi).first;
				pCmi = &(*I);
			}
			InsertOrUpdateCollectionItem(*pCmi, vt.second);
		}

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not add collections items");
	}
}

void PocoDatabaseImpl::DeleteCollectionsItems (const ptime& timestamp, const Collections& collections)
{
	try
	{
		DBTransactor transactor(m_pSession);

		const string& t = lexical_cast<string>(timestamp);
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			if (collections.find(cmi.m_collection) != collections.end())
				(*m_pSession) << "delete from " << cmi.m_tblname << " where t='" << t << "';", now;
		}

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not delete data");
	}
}

void PocoDatabaseImpl::GenerateExpressions (Expressions& expressions, const ptime& timestamp) const throw (Xception)
{
	try
	{
		typedef multimap<ptime, pair<any, string> > Data;

		Data data;
		size_t n = 0;
		{
			foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
			{
				// build the SQL string
				string sql("select t,");
				switch (cmi.m_type)
				{
					case 'r':	sql += "n,d"; break;
					default:	sql += "v"; break;
				}
				sql += " from " + cmi.m_tblname;
				if (timestamp != not_a_date_time)
					sql += " where t='" + lexical_cast<string>(timestamp) + "'";
				sql += " order by 1;";

				// execute
				Statement select(*m_pSession);
				select << sql;
				select.execute();

				// fetch data
				RecordSet rs(select);
				if (rs.moveFirst())
				{
					do {
						pair<any, string> value;
						switch (cmi.m_type)
						{
							case 'n':	value.first = rs.value(1).convert<double>(); break;
							case 'r':	value.first = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
							default:	value.first = rs.value(1).convert<string>(); break;
						}
						value.second = cmi.m_collection;
						data.insert(Data::value_type(lexical_cast<ptime>(rs.value(0).convert<string>()), value));
					} while (rs.moveNext());
				}
				n++;
			}
		}

		Data::const_iterator I = data.begin();
		while (I != data.end())
		{
			const ptime timestamp = (*I).first;

			string expr;
			string comment;
			Data::const_iterator J = data.upper_bound(timestamp);
			while (I != J)
			{
				const Data::value_type& vt = *I++;
				const any& a = vt.second.first;
				switch (GetCollectionTypeFromValue(a))
				{
					case 'n':	expr += (format(" %g") % any_cast<double>(a)).str() + vt.second.second; break;
					case 'r':	expr += (format(" %g") % any_cast<Ratio>(a).m_numerator).str() + vt.second.second + (format("%g") % any_cast<Ratio>(a).m_denominator).str(); break;
					case 't':	comment = any_cast<string>(a); break;
					default:	break;
				}
			}
			if (expr[0] == ' ')
				expr.erase(0, 1);
			if (!comment.empty())
				expr += " ; " + comment;

			expressions.insert(Database::Expressions::value_type(timestamp, expr));
		}
	}
	catch (...)
	{
		throw Xception("error selecting data");
	}
}
