// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "../libpdrx/build.h"
#include "db.h"
#include "out.h"

//=== ConfigImpl ===========================================================
class ConfigImpl: public Config {

	public:

	ConfigImpl ();
};

ConfigImpl::ConfigImpl ()
	: Config()
{
	m_od1.add_options()

		// common options
		("help,?",			"show this help screen and quit")
		("version,V",			"display version number and quit")
		("verbose,v",			"produce verbose output on stdout")
		("config,c", value<string>(),	"use specific configuration file")
		("now,n", value<string>(),	"define what is \"now\", arg is CCYY-MM-DD[-hh:mm[:ss]]")
		("interactive,i",		"don't generate any output, start interactive")
		("fast,f",			"cache internal informations in memory, needs much more RAM")
	;

	// (we need a named option for the sum of all the command line
	// tokens to make an expression from it but we don't want to have
	// this on the help screen)
	m_od2.add_options()

		("expression", value<vector<string> >(),	"")
		("cmd_line_expr", value<vector<string> >(),	"")
	;

	m_pod.add("cmd_line_expr", 100);
}

//=== main =================================================================
int main (int argc, char* argv[])
{
	try
	{
		InitEncodings();

		// parse the command line and the config file
		auto_ptr<Config> config(ConfigFactory<ConfigImpl>::Create(argc, argv));

		// handle the very simple cases
		if (config->GetBoolOption("version"))
		{
			cout	<< "pdx - personal data expert, " << VERSION << endl
				<< "(C) 2010 Torsten Mueller, Bern, Switzerland" << endl
				<< '(' << BUILD << ')' << endl
				<< "This program is free software under the terms of the GNU General Public License v2 or later." << endl;
			return 0;
		}

		if (config->GetBoolOption("help"))
		{
			cout << "pdx - personal data expert" << endl;
			cout << "usage: pdr [options] [args]" << endl;
			config->Help(cout);
			return 0;
		}

		// not very simple, work with the database
		{
			auto_ptr<Database> database(DBFactory::Create(*config));
			OutputFactory factory;

			if (config->GetBoolOption("interactive"))
				auto_ptr<Output>(factory.GetOutputInteractive())->Do(*config, *database);
			else
			{
				// create output
				bool verbose = config->GetBoolOption("verbose");
				if (verbose)
					cout << "generating output ";

				const OutputFactory::Outputs& outputs = factory.GetOutputs(*config);
				foreach (const Output* pOutput, outputs)
				{
					if (verbose)
						cout << ':' << flush;
					try
					{
						pOutput->Do(*config, *database);
					}
					catch (...)
					{
						if (verbose)
							cout << endl;
						throw;
					}
				}
				if (verbose)
					cout << endl;
			}
		}

		return 0;
	}
	catch (const Xception& e)
	{
		cerr << "*** fatal: " << e.Message() << endl;
		return 1;
	}
	catch (...)
	{
		cerr << "*** fatal: internal error" << endl;
		return 1;
	}
}
