// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

//=== PocoDatabaseImpl (abstract base class) ===============================
PocoDatabaseImpl::PocoDatabaseImpl (const string& connect, bool verbose, bool fast, const string& KEY, Connector* pConnector)
	: PocoDatabase(connect, verbose, KEY, pConnector)
	, m_fast(fast)
	, m_selections()
{
}

PocoDatabaseImpl::~PocoDatabaseImpl ()
{
}

void PocoDatabaseImpl::Connect () throw (Xception)
{
	return PocoDatabase::Connect();
}

	inline string selection_key (const string& collection, const ptime& begin, const ptime& end)
	{
		return collection + '|' + lexical_cast<string>(begin) + '|' + lexical_cast<string>(end);
	}

void PocoDatabaseImpl::Select (shared_ptr<Selection>& selection, const string& collection, const ptime& begin, const ptime& end) const throw (Xception)
{
	if (m_fast)
	{
		Selections::const_iterator I = m_selections.find(selection_key(collection, begin, end));
		if (I != m_selections.end())
		{
			selection = (*I).second; // just return what we know
			return;
		}
	}

	selection->resize(extents[0][2]); // clear any contents

	const CollectionMetaInfo& cmi = GetCollectionMetaInfo(collection);

	try
	{
		// build the SQL string
		string sql("select t,");
		switch (cmi.m_type)
		{
			case 'r':	sql += "n,d"; break;
			default:	sql += "v"; break;
		}
		sql += " from " + cmi.m_tblname;

		string where;
		{
			if (begin != not_a_date_time)
				where += "t>='" + lexical_cast<string, ptime>(begin) + "'";

			if (end != not_a_date_time)
			{
				if (!where.empty())
					where += " and ";
				where += "t<'" + lexical_cast<string, ptime>(end) + "'";
			}
		}
		if (!where.empty())
			sql += " where " + where;

		sql += " order by 1;";

		// execute the statement
		Statement select(*m_pSession);
		select << sql;
		select.execute();

		// fetch data
		RecordSet rs(select);
		if (rs.moveFirst())
		{
			size_t row = 0;
			do {
				selection->resize(extents[row + 1][2]);

				// we return datetime values as string,
				// this is more handy for later grouping
				// during aggregation
				(*selection)[row][0] = rs.value(0).convert<string>();

				switch (cmi.m_type)
				{
					case 'n':	(*selection)[row][1] = rs.value(1).convert<double>(); break;
					case 'r':	(*selection)[row][1] = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
					default:	(*selection)[row][1] = rs.value(1).convert<string>(); break;
				}

				row++;

			} while (rs.moveNext());
		}

		if (m_fast)
			m_selections.insert(Selections::value_type(selection_key(collection, begin, end), selection));
	}
	catch (...)
	{
		throw Xception("error selecting data");
	}
}

void PocoDatabaseImpl::NoSchemaCallback () throw (Xception)
{
	throw Xception("no schema found in database");
}
