// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "diagram_impl.h"

//=== DiagramImpl ==========================================================
DiagramImpl::DiagramImpl ()
	: m_xmin(not_a_date_time)
	, m_xmax(not_a_date_time)
	, m_xunit((XUnit)-1)
	, m_ymin(0.0)
	, m_ymax(0.0)
	, m_yunit(0.0)
	, m_increment(not_a_date_time)
{
}

void DiagramImpl::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color)
{
	m_xmin = x_min;
	m_xmax = x_max;
	m_xunit = x_unit;
	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;

	switch (m_xunit)
	{
		case years:	m_increment = posix_time::hours(365 * 24); break;
		case months:	m_increment = posix_time::hours(30 * 24); break;
		case days:	m_increment = posix_time::hours(24); break;
		case hours:	m_increment = posix_time::hours(1); break;
		case minutes:	m_increment = posix_time::minutes(1); break;
		default:	m_increment = posix_time::seconds(1); break;
	}
}

void DiagramImpl::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color)
{
	// for folding we must handle dates in the year 9999, this is a bit
	// special but indeed no problem, we just need an according axis
	switch (fi)
	{
		case year:
		{
			m_xmin = ptime(date(9999,  Jan, 1), time_duration(0, 0, 0));
			m_xmax = ptime(date(10000, Jan, 1), time_duration(0, 0, 0));
			m_xunit = months;
			m_increment = posix_time::hours(30 * 24);
			break;
		}
		case month:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Feb,  1), time_duration(0, 0, 0));
			m_xunit = days;
			m_increment = posix_time::hours(24);
			break;
		}
		case day:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  2), time_duration(0, 0, 0));
			m_xunit = hours;
			m_increment = posix_time::hours(1);
			break;
		}
		case hour:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(1, 0, 0));
			m_xunit = minutes;
			m_increment = posix_time::minutes(1);
			break;
		}
		case minute:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(0, 1, 0));
			m_xunit = seconds;
			m_increment = posix_time::seconds(1);
			break;
		}
	}

	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;
}

string DiagramImpl::Fold (const string& timestamp, Diagram::FoldInterval fi)
{
	string result(timestamp);
	switch (fi)
	{
		case Diagram::year:	result.erase(0,  5); result.erase(2); break;
		case Diagram::month:	result.erase(0,  8); result.erase(2); break;
		case Diagram::day:	result.erase(0, 11); result.erase(2); break;
		case Diagram::hour:	result.erase(0, 14); result.erase(2); break;
		case Diagram::minute:	result.erase(0, 17); break;
	}
	return result;
}

string DiagramImpl::FormatY () const
{
	stringstream ss;
	ss << m_yunit;
	string s(ss.str());
	string::size_type pos = s.find('.');
	if (pos != string::npos)
	{
		s.erase(0, pos + 1);
		return (format("%%1.%df") % s.length()).str();
	}
	else
		return "%1.0f";
}
