// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "diagram_impl.h"

#ifdef USE_BOARD

using namespace LibBoard;

#define	COLOR(rgb)	Color(((rgb >> 16) & 0x0FF), ((rgb >> 8) & 0x0FF), (rgb & 0x0FF))

#define	LINE_THICKNESS	0.25

//=== DiagramSVG ===========================================================
DiagramSVG::DiagramSVG ()
	: DiagramImpl()
	, m_canvas()
	, m_shapes()
{
	m_diagram.width = m_diagram.height = 0.0;

//#ifdef _BOARD_PATH_H_
	// libboard 0.9.0
//	m_canvas.setUnit(Board::UPoint);
//#else
	// libboard 0.8.8
	m_canvas.setUnit(Board::PT);
//#endif
}

void DiagramSVG::OpenCanvas (size_t width, size_t height, RGB bgColor)
{
	// - set a size,
	// - fill the background

	m_diagram.width = width;
	m_diagram.height = height;

	m_canvas.clear(COLOR(bgColor));
}

void DiagramSVG::CloseCanvas ()
{
	m_canvas << m_shapes;
}

void DiagramSVG::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color)
{
	DiagramImpl::DrawAxes(x_min, x_max, x_unit, y_min, y_max, y_unit, color);

	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions
	const double char_width = m_diagram.width / 94.0;
	const double char_height = m_diagram.width / 94.0;
	const float font_size = float(m_diagram.width / 60.0);

	// Step 1: x-axis
	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			s1.erase(7);
			s2 = s1;
			s1 += "-01 00:00:00.000";
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(color));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
			}
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			switch (m_xunit)
			{
				case years:	s1.erase(4);  s2 = s1; s1 += "-01-01 00:00:00.000"; break;
				case days:	s1.erase(10); s2 = s1; s1 += " 00:00:00.000"; break;
				case hours:	s1.erase(13); s2 = s1; s1 += ":00:00.000"; break;
				case minutes:	s1.erase(16); s2 = s1; s1 += ":00.000"; break;
				default:	break;
			}
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(color));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
			}
		}
	}

	// Step 2: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, font_size, COLOR(color));
	}
}

void DiagramSVG::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color)
{
	DiagramImpl::DrawAxes(fi, y_min, y_max, y_unit, color);

	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions
	const double char_width = m_diagram.width / 94.0;
	const double char_height = m_diagram.width / 94.0;
	const float font_size = float(m_diagram.width / 60.0);

	// Step 1: x-axis
	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
		}
	}

	// Step 2: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, font_size, COLOR(color));
	}
}

void DiagramSVG::DrawCurve (const Selection& s, RGB color, LineStyle style, double thickness, bool continuous, int bar1, int bar2)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	struct Dot {
		ptime	m_timestamp;
		any	m_value;
		Dot ()
			: m_timestamp(not_a_date_time)
			, m_value()
		{
		}
		Dot (const string& timestamp, const any& value)
			: m_timestamp(Diagram::CompletePtimeFromString(timestamp))
			, m_value(value)
		{
		}
		operator bool () const
		{
			return !m_value.empty();
		}
	};

	if (thickness == 0.0)
		thickness = LINE_THICKNESS;
	else
	{
		// we want a thickness of 1.0 to draw a line with about
		// 1 pixel width, so we must adjust this a bit because
		// libboard draws already a really thick line using a
		// thickness of 1.0
		thickness /= 2.0;
	}

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		switch (style)
		{
			case bars:
			{
				if (current)
				{
					ptime t(current.m_timestamp);
					double x = CalcX(t);
					double y = CalcY(any_cast<double>(current.m_value));
					double x_width;
					if (bar1 != 0 && bar2 != 0 && bar1 <= bar2)
					{
						x_width = (CalcX(t + m_increment) - x) / double(bar2);
						x += x_width * double(bar1 - 1);
					}
					else
						x_width = CalcX(t + m_increment) - x;
					if (x >= 0)
						m_shapes << LibBoard::Rectangle(x, y, x_width, y - CalcY(m_ymin), COLOR(color), COLOR(color), 0);
				}
				break;
			}
			case symbol_plus:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_vbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_hbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_cross:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
					m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_circle:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Circle(x, y, extent, COLOR(color), Color(false), (float)thickness);
				}
				break;
			}
			default: // zick-zack
			{
				if (current.m_timestamp < m_xmin)
					continue;
				if (last && current)
				{
					m_shapes << Line(
						CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)),
						CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)),
						COLOR(color), (float)thickness
					);
				}
				break;
			}
		}

		if (current || !continuous)
			last = current;
	}
}

void DiagramSVG::DrawVLine (const ptime& timestamp, RGB color, double thickness)
{
	double x = CalcX(timestamp);
	m_shapes << Line(x, CalcY(m_ymin), x, CalcY(m_ymax), COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
}

void DiagramSVG::DrawVLine (const time_duration& time, RGB color, double thickness)
{
	DrawVLine(ptime(date(9999, 1, 1), time), color, thickness);
}

void DiagramSVG::DrawHLine (double value, RGB color, double thickness)
{
	double y = CalcY(value);
	m_shapes << Line(CalcX(m_xmin), y, m_diagram.width, y, COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
}

void DiagramSVG::SaveAs (const string& filename) const
{
	m_canvas.saveSVG(filename.c_str(), Board::BoundingBox);
}

double DiagramSVG::CalcX (const ptime& x) const
{
	double a = (x - m_xmin).total_seconds();
	double b = (m_xmax - m_xmin).total_seconds();
	return (a / b) * m_diagram.width;
}

double DiagramSVG::CalcY (double y) const
{
	double a = y - m_ymin;
	double b = m_ymax - m_ymin;
	return (a / b) * m_diagram.height;
}

#endif // USE_BOARD
