// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "db.h"
#include "out_impl.h"
#include "out_ftree.h"

//=== OutputReport =========================================================
OutputReport::OutputReport (const string& option_key)
	: OutputImpl(option_key)
{
}

void OutputReport::Do (const Config& config, const Database& database) const throw (Xception)
{
	// get some configuration data
	string comment_begin = config.GetStringOption(m_option_key + ".comment_begin");
	if (comment_begin.empty())
		comment_begin = "<!--";

	string comment_end = config.GetStringOption(m_option_key + ".comment_end");
	if (comment_end.empty())
		comment_end = "-->";

	const string& input_file = config.GetFilenameOption(m_option_key + ".input_file");
	if (input_file.empty())
		throw Xception(format("missing input_file specification in report configuration: %s") % m_option_key);

	const string& output_file = config.GetFilenameOption(m_option_key + ".output_file");

	string encoding = config.GetStringOption(m_option_key + ".encoding");
	if (encoding.empty())
		encoding = GetEncoding().canonicalName();

	// step 1: read input file into blocks of text, the blocks are
	// separated by comment_begin and comment_end
	typedef vector<string> Blocks;
	Blocks blocks;
	{
		ifstream is(input_file.c_str(), ios::in);
		if (!is.good())
			throw Xception(format("cannot open input file: %s") % input_file);

		set<size_t> positions;
		positions.insert(0);
		size_t pos;
		string line;
		while (pos = is.tellg() , getline(is, line))	// hey, comma operator!
		{
			string::size_type p = 0;
			while (p < line.length())
			{
				p = line.find(comment_begin, p);
				if (p == string::npos)
					break;
				positions.insert(pos + p);	// point onto the begin
				p++;
			}

			p = 0;
			while (p < line.length())
			{
				p = line.find(comment_end, p);
				if (p == string::npos)
					break;
				p += comment_end.length();
				positions.insert(pos + p);	// point one character behind the end
			}
		}
		positions.insert(pos);
		is.clear();

		is.seekg(0);
		set<size_t>::const_iterator I = positions.begin();
		while (I != positions.end())
		{
			size_t begin = *I++;
			if (I == positions.end())
				break;
			size_t end = *I;
			size_t len = end - begin;
			char* buf = new char[len];
			is.read(buf, len);
			blocks.push_back(ConvertFrom(string(buf, len), GetEncoding(encoding)));
			delete[] buf;
		}
	}

	// step 2: process the blocks containing report definitions
	bool verbose = config.GetBoolOption("verbose");
	foreach (string& block, blocks)
	{
		if (block.find(comment_begin) == 0)
		{
			string s(block);
			s.erase(0, comment_begin.length());
			s.erase(s.length() - comment_end.length());
			trim(s);

			if (verbose)
				cout << '.' << flush;

			auto_ptr<FTree> ft(Parse(s));
			stringstream ss;
			FContext context(config, database);
			ss << ft->Evaluate(context);
			block = ss.str();
		}
	}

	// step 3: write output
	{
		ostream* pos = NULL;
		auto_ptr<ofstream> apos;
		if (output_file.empty())
			pos = &cout;
		else
		{
			apos = auto_ptr<ofstream>(new ofstream(output_file.c_str(), ios::out));
			pos = apos.get();
		}
		ostream& os = *pos;
		if (!os.good())
			throw Xception(format("cannot open output file: %s") % output_file);

		foreach (const string& block, blocks)
		{
			os << ConvertTo(block, GetEncoding(encoding));
		}
	}
}
