// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/encoding.h"
#include "config_impl.h"
#include "diagram_impl.h"

#ifdef USE_BOARD

using namespace LibBoard;

#define	COLOR(rgb)	Color(((rgb >> 16) & 0x0FF), ((rgb >> 8) & 0x0FF), (rgb & 0x0FF))

#define	LINE_THICKNESS	0.25

//=== BoardDiagram =========================================================
BoardDiagram::BoardDiagram ()
	: DiagramImpl()
	, m_canvas()
	, m_shapes()
{
	m_diagram.width = m_diagram.height = 0.0;
	m_canvas.setUnit(Board::UPoint);
}

void BoardDiagram::OpenCanvas (size_t width, size_t height, RGB bgColor)
{
	// - set a size,
	// - fill the background

	m_diagram.width = width;
	m_diagram.height = height;

	m_canvas.clear(COLOR(bgColor));
}

void BoardDiagram::CloseCanvas ()
{
	m_canvas << m_shapes;
}

void BoardDiagram::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color, const ConfigImpl::Periods& periods)
{
	DiagramImpl::DrawAxes(x_min, x_max, x_unit, y_min, y_max, y_unit, color, periods);

	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions
	const double char_width = m_diagram.width / 94.0;
	const double char_height = m_diagram.width / 94.0;
	const float font_size = float(m_diagram.width / 60.0);

	// Step 1: iterate over the time periods, make backgrounds
	foreach(const ConfigImpl::Period* pPeriod, periods)
	{
		ptime begin(not_a_date_time), end(not_a_date_time);

		{
			ptime t(pPeriod->m_begin, time_duration(0, 0, 0, 0));
			if (t < x_min)
				begin = x_min;
			else
			{
				if (t > x_max)
					begin = x_max;
				else
					begin = t;
			}
		}

		{
			ptime t(pPeriod->m_end, time_duration(0, 0, 0, 0));
			if (t < x_min)
				end = x_min;
			else
			{
				if (t > x_max)
					end = x_max;
				else
					end = t;
			}
		}

		if (begin != end)
			m_shapes << LibBoard::Rectangle(CalcX(begin), CalcY(y_max), CalcX(end) - CalcX(begin), CalcY(y_max) - CalcY(y_min), COLOR(pPeriod->m_color), COLOR(pPeriod->m_color), 0.0);
	}

	// Step 2: x-axis
	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			s1.erase(7);
			s2 = s1;
			s1 += "-01 00:00:00.000";
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(color));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
			}
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			switch (m_xunit)
			{
				case years:	s1.erase(4);  s2 = s1; s1 += "-01-01 00:00:00.000"; break;
				case days:	s1.erase(10); s2 = s1; s1 += " 00:00:00.000"; break;
				case hours:	s1.erase(13); s2 = s1; s1 += ":00:00.000"; break;
				case minutes:	s1.erase(16); s2 = s1; s1 += ":00.000"; break;
				default:	break;
			}
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				RGB c = (m_sundays != (RGB)-1 && m_xunit == days && t.date().day_of_week() == 0) ? m_sundays : color;
				Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(c));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
			}
		}
	}

	// Step 3: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, font_size, COLOR(color));
	}
}

void BoardDiagram::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color)
{
	DiagramImpl::DrawAxes(fi, y_min, y_max, y_unit, color);

	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions
	const double char_width = m_diagram.width / 94.0;
	const double char_height = m_diagram.width / 94.0;
	const float font_size = float(m_diagram.width / 60.0);

	// Step 1: x-axis
	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
		}
	}

	// Step 2: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, font_size, COLOR(color));
	}
}

void BoardDiagram::DrawCurve (const Selection& s, RGB color, LineStyle style, double thickness, bool continuous)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	if (thickness == 0.0)
		thickness = LINE_THICKNESS;
	else
	{
		// we want a thickness of 1.0 to draw a line with about
		// 1 pixel width, so we must adjust this a bit because
		// libboard draws already a really thick line using a
		// thickness of 1.0
		thickness /= 2.0;
	}

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		switch (style)
		{
			case symbol_dot:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					m_shapes << LibBoard::Dot(x, y, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_plus:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_vbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_hbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_x:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 200.0; // half a percent of m_diagram.width
					m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
					m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_X:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
					m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_circle:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << Circle(x, y, extent, COLOR(color), Color(false), (float)thickness);
				}
				break;
			}
			case symbol_square:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					m_shapes << LibBoard::Rectangle(x - extent / 2.0, y - extent / 2.0, extent, extent, COLOR(color), Color(false), (float)thickness);
				}
				break;
			}
			default: // zick-zack
			{
				if (current.m_timestamp < m_xmin)
					continue;
				if (last && current)
				{
					m_shapes << Line(
						CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)),
						CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)),
						COLOR(color), (float)thickness
					);
				}
				break;
			}
		}

		if (current || !continuous)
			last = current;
	}
}

void BoardDiagram::DrawStairs (const Selection& s, RGB color, double thickness)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	if (thickness == 0.0)
		thickness = LINE_THICKNESS;
	else
	{
		// we want a thickness of 1.0 to draw a line with about
		// 1 pixel width, so we must adjust this a bit because
		// libboard draws already a really thick line using a
		// thickness of 1.0
		thickness /= 2.0;
	}

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		if (current.m_timestamp < m_xmin)
			continue;
		if (last && current)
		{
			m_shapes << Line(
				CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)),
				CalcX(current.m_timestamp), CalcY(any_cast<double>(last.m_value)),
				COLOR(color), (float)thickness
			);
			m_shapes << Line(
				CalcX(current.m_timestamp), CalcY(any_cast<double>(last.m_value)),
				CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)),
				COLOR(color), (float)thickness
			);
		}

		if (current)
			last = current;
	}
}

void BoardDiagram::DrawVLine (const ptime& timestamp, RGB color, double thickness)
{
	double x = CalcX(timestamp);
	m_shapes << Line(x, CalcY(m_ymin), x, CalcY(m_ymax), COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
}

void BoardDiagram::DrawVLine (const time_duration& time, RGB color, double thickness)
{
	DrawVLine(ptime(date(9999, 1, 1), time), color, thickness);
}

void BoardDiagram::DrawHLine (double value, RGB color, double thickness)
{
	double y = CalcY(value);
	m_shapes << Line(CalcX(m_xmin), y, m_diagram.width, y, COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
}

void BoardDiagram::DrawBar (const ptime& t, double value, int bar1, int bar2, RGB color, bool fix)
{
	double x = CalcX(t);
	double y = CalcY(value);
	double x_width = (fix) ? (CalcX(t + m_increment) - x) / double(bar2) : (CalcX(min(t + m_increment, m_xmax)) - x) / double(bar2);
	x += x_width * double(bar1 - 1);

	if (x >= 0)
		m_shapes << LibBoard::Rectangle(x, y, x_width, y - CalcY(m_ymin), COLOR(color), COLOR(color), 0);
}

double BoardDiagram::CalcX (const ptime& x) const
{
	double a = (x - m_xmin).total_seconds();
	double b = (m_xmax - m_xmin).total_seconds();
	return (a / b) * m_diagram.width;
}

double BoardDiagram::CalcY (double y) const
{
	double a = y - m_ymin;
	double b = m_ymax - m_ymin;
	return (a / b) * m_diagram.height;
}

//=== DiagramSVG ===========================================================
DiagramSVG::DiagramSVG ()
	: BoardDiagram()
{
}

void DiagramSVG::SaveAs (const string& filename) const
{
	m_canvas.saveSVG(filename.c_str(), Board::BoundingBox);
}

//=== DiagramEPS ===========================================================
DiagramEPS::DiagramEPS ()
	: BoardDiagram()
{
}

void DiagramEPS::SaveAs (const string& filename) const
{
	m_canvas.saveEPS(filename.c_str(), Board::BoundingBox);
}

//=== DiagramXFig ==========================================================
DiagramXFig::DiagramXFig ()
	: BoardDiagram()
{
}

void DiagramXFig::SaveAs (const string& filename) const
{
	m_canvas.saveFIG(filename.c_str(), Board::BoundingBox);
}

#endif // USE_BOARD
