// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "in_impl.h"

#include <Poco/Net/NetException.h>
#include <Poco/Net/POP3ClientSession.h>
#include <Poco/Net/MailMessage.h>
#include <Poco/Net/SecureStreamSocket.h>
#include <Poco/Net/SSLManager.h>
#include <Poco/Net/ConsoleCertificateHandler.h>
#include <Poco/SharedPtr.h>

//=== Pop3MailClient =======================================================
Pop3MailClient::Pop3MailClient (const string& option_key)
	: InputImpl(option_key)
{
}

	class SSLInitializer
	{
		public:

		SSLInitializer()
		{
			Poco::Net::initializeSSL();
		}

		~SSLInitializer()
		{
			Poco::Net::uninitializeSSL();
		}
	};

	static string extract_charset_from_content_type (const string& content_type)
	{
		// a content type is normally something like this:
		//
		//	text/plain; charset=iso-8859-15
		//
		// the part in front of the semicolon defines the type of the
		// content, we only accept text/plain here be definition, the
		// other part defines the character set of the text
		//
		// some mail clients insert newlines after the semicolon or
		// spaces or tabs at some places in this string, so we have to be
		// very robust here

		string s(content_type);
		to_lower(s);

		for (size_t i = 0; i < s.length(); i++)
		{
			if (s[i] < 0x20)
				s[i] = 0x20;
		}

		static const regex rx_charset("^text/plain\\s*;\\s*charset\\s*=\\s*(.+)\\s*$");
		smatch mr;
		if (regex_match(s, mr, rx_charset, boost::match_not_dot_newline))
		{
			string charset(mr[1]);
			string c(charset);
			to_lower(c);
			if (c == "us-ascii")
				charset = "ASCII";
			return charset;
		}
		else
			return string(); // no charset or not text/plain
	}

void Pop3MailClient::Do (const Config& config, Database& database) const throw (Xception)
{
	try
	{
		bool verbose = config.GetBoolOption("verbose");
		if (verbose)
			encoded::cout << "looking for mail (POP3)" << endl;

		// get configuration data
		const string& server = config.GetStringOption(m_option_key + ".server");
		if (server.empty())
			THROW(format("missing specification in configuration file: %s.server)") % m_option_key);

		Poco::UInt16 port = (config.HasOption(m_option_key + ".port"))
			? (Poco::UInt16)config.GetDoubleOption(m_option_key + ".port")
			: 110;

		const string& account = config.GetStringOption(m_option_key + ".account");
		if (account.empty())
			THROW(format("missing specification in configuration file: %s.account)") % m_option_key);

		const string& password = config.GetStringOption(m_option_key + ".password");
		const string& subject = config.GetStringOption(m_option_key + ".subject");
		bool keep = config.GetBoolOption(m_option_key + ".keep");

		// open connection
		auto_ptr<SSLInitializer> pSSLInitializer;
		auto_ptr<Poco::Net::SecureStreamSocket> pSSS;
		auto_ptr<Poco::Net::POP3ClientSession> pSession;
		if (port == 995)
		{
			// this magic stuff is from a Poco sample //////////
			pSSLInitializer = auto_ptr<SSLInitializer>(new SSLInitializer());

			Poco::SharedPtr<Poco::Net::InvalidCertificateHandler> pCert = new Poco::Net::ConsoleCertificateHandler(false);
			Poco::Net::Context::Ptr pContext = new Poco::Net::Context(Poco::Net::Context::CLIENT_USE, "", "", "",
				 Poco::Net::Context::VERIFY_RELAXED, 9, true, "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH");
			Poco::Net::SSLManager::instance().initializeClient(0, pCert, pContext);
			////////////////////////////////////////////////////

			// now we can open a secure socket ...
			pSSS = auto_ptr<Poco::Net::SecureStreamSocket>(new Poco::Net::SecureStreamSocket(Poco::Net::SocketAddress(server, port)));

			// ... and a session upon the socket as well
			pSession = auto_ptr<Poco::Net::POP3ClientSession>(new Poco::Net::POP3ClientSession(*pSSS));
		}
		else
			pSession = auto_ptr<Poco::Net::POP3ClientSession>(new Poco::Net::POP3ClientSession(server));

		// we authenticate using a non-encrypted password
		pSession->login(account, password);

		// process available mails
		Poco::Net::POP3ClientSession::MessageInfoVec messages;
		pSession->listMessages(messages);

		Database::Collections collections;
		database.GetCollections(collections);

		bool data = false;
		bool rejects = false;
		Poco::Net::POP3ClientSession::MessageInfoVec::const_iterator I = messages.begin();
		while (I != messages.end())
		{
			int id = (*I++).id;

			Poco::Net::MailMessage message;
			pSession->retrieveMessage(id, message);

			// check subject
			string s(message["Subject"]);
			{
				trim(s);
				s = InputImpl::ConvertMailSubject(s);
			}

			if (s != subject)
				continue;

			data = true;

			// extract timestamp
			ptime timestamp(not_a_date_time);
			{
				string d(message["Date"]);
				trim(d);
				timestamp = lexical_cast_mime_date(d);
			}

			// get message body and decode it correctly (this means make it UTF-8)
			const string& charset = extract_charset_from_content_type(message.getContentType());
			stringstream ss;
			if (charset.empty())
				ss << message.getContent(); // should be 7bit ASCII
			else
				ss << Decode(message.getContent(), SpecificEncoding(charset));

			// parse line by line, every line an expression
			string line;
			while (getline(ss, line))
			{
				trim(line);
				if (line.empty())
					continue;

				try
				{
					Database::CollectionsItems items;
					Parse(line, timestamp, verbose, collections, items);
					database.AddCollectionsItems(items);
				}
				catch (const Xception& x)
				{
					encoded::cerr << x.Message(Xception::brief) << endl;
					database.AddRejected(timestamp, line);
					rejects = true;
				}
			}

			// delete if neccessary
			if (!keep)
				pSession->deleteMessage(id);
		}
		if (verbose && !data)
			encoded::cout << "    no data on server" << endl;

		// close connection
		pSession->close();

		if (rejects)
			encoded::cerr << "!!! at least one expression has been rejected, try -r to list rejections !!!" << endl;
	}
	CATCH_RETHROW("could not retrieve mail data from POP3 server")
}
