/*
 * pdsbvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <stdlib.h>
#include <math.h>
#include <pds/pdsbvector.h>



////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsBVector                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsBVector *pds_bvector_new(PdsBaNatural Nel)
 *  \brief Crea una matriz de tipo PdsBVector.
 *  \param[in] Nel Es el número de bits del vector.
 *  \return Un puntero al vector de tipo PdsBVector, o NULL si hubo un error.
 *  \ingroup PdsBVectorGroup
 */
PdsBVector *pds_bvector_new(PdsBaNatural Nel)
{
	PdsBVector *BVec=NULL;
	
	BVec=(PdsBVector*)calloc(1,sizeof(PdsBVector));
	if(BVec==NULL)	return NULL;

	BVec->Nel=Nel;
	BVec->Nbytes=(Nel+7)/8;

	BVec->V=(PdsBaByte *)calloc(BVec->Nbytes,sizeof(PdsBaByte));
	if(BVec->V==NULL)
	{
		free(BVec);
		return NULL;
	}
	
	return BVec;
}


/** \fn int pds_bvector_init(PdsBVector *BVector, PdsBaBit Bit)
 *  \brief Inicia con el valor Bit los Nel bits del vector.
 *  \param[in,out] BVector Es el vector a inicializar.
 *  \param[in] Bit Es el valor de los bits del vector, solo {0,1}.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL o Bit!={0,1}.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_init(PdsBVector *BVector, PdsBaBit Bit)
{
	PdsBaNatural i,j;
	PdsBaByte Byte;
	PdsBaByte k;

	if(BVector==NULL)	return FALSE;
	if(Bit>1)		return FALSE;

	k=(PdsBaByte)(BVector->Nbytes*8-BVector->Nel);

	Byte=0;
	for(j=0;j<8;j++)
	{
		Byte=Byte|(Bit<<j);
	}

	for(i=0;i<BVector->Nbytes;i++)
	{
		BVector->V[i]=Byte;
	}

	for(j=0;j<k;j++)
	{
		BVector->V[BVector->Nbytes-1]=BVector->V[BVector->Nbytes-1]&(~(1<<(7-j)));
	}
	
	return TRUE;
}


/** \fn int pds_bvector_init_rand(PdsBVector *BVector, PdsBaReal p)
 *  \brief Inicia el vector con algunos "1"s, cada bit tiene una probabilidad p de ser "1" (P{V_i=1}=p).
 *  usa la funcion rand de la biblioteca standar de c.
 *  \param[in,out] BVector Es el vector a inicializar.
 *  \param[in] p Es la probabilidad de cada bit ser 1.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, p<0 o p>1.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_init_rand(PdsBVector *BVector, PdsBaReal p)
{
	PdsBaNatural i,j,k;
	PdsBaByte Byte;


	if(BVector==NULL)	return FALSE;
	if((p<0.0)||(p>1.0))	return FALSE;

	for(i=0,k=0;i<BVector->Nbytes;i++)
	{
		Byte=0;
		for(j=0;(j<8)&&(k<BVector->Nel);j++,k++)
		{
			if(((rand()*1.0)/RAND_MAX)<=p)
			{
				Byte=Byte|(1<<j);
			}
			
		}
		BVector->V[i]=Byte;
	}
	
	return TRUE;
}


/** \fn int pds_bvector_cmp(const PdsBVector *BVector1,const PdsBVector *BVector2, PdsBaNatural *n)
 *  \brief Encuentra el número de elementos distintos entre los dos vectores.
 *  \param[in] BVector1 Es uno de los vectores a comparar.
 *  \param[in] BVector2 Es uno de los vectores a comparar.
 *  \param[in] n Es el número de elementos distintos entre los dos vectores.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_cmp(const PdsBVector *BVector1,const PdsBVector *BVector2, PdsBaNatural *n)
{
	PdsBaNatural i,j,k;
	PdsBaByte Byte;

	*n=0;

	if(BVector1==NULL)			return FALSE;
	if(BVector2==NULL)			return FALSE;
	if(BVector1->Nel!=BVector2->Nel)	return FALSE;	
	if(BVector1->Nbytes!=BVector2->Nbytes)	return FALSE;	

	for(i=0,k=0;i<BVector1->Nbytes;i++)
	{
		Byte=BVector1->V[i]^BVector2->V[i];
		for(j=0;(j<8)&&(k<BVector2->Nel);j++,k++)
		{
			*n=*n+((Byte&(1<<j))>>j);
		}
	}
	return TRUE;
}


/** \fn int pds_bvector_cp(PdsBVector *BVector1,const PdsBVector *BVector2)
 *  \brief Copia el contenido del vector BVector2 a BVector1, ambos vectores 
 *  deben tener el mismo tamaño.
 *  \param[in,out] BVector1 Es el vector en donde se copiará.
 *  \param[in] BVector2 Es el vector a copiar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_cp(PdsBVector *BVector1,const PdsBVector *BVector2)
{
	PdsBaNatural i;

	if(BVector1==NULL)			return FALSE;
	if(BVector2==NULL)			return FALSE;
	if(BVector1->Nel!=BVector2->Nel)	return FALSE;	
	if(BVector1->Nbytes!=BVector2->Nbytes)	return FALSE;	

	for(i=0;i<BVector1->Nbytes;i++)
	{
		BVector1->V[i]=BVector2->V[i];
	}
	return TRUE;
}


/** \fn int pds_bvector_set_natural(PdsBVector *BVector,PdsBaNatural n)
 *  \brief Escribe un número natural codificado en binario, en el vector BVector.
 *  Si el número en binario es mayor que el vector no dará error, simplemente se
 *  copiará hasta donde pueda.
 *  \param[in,out] BVector Es el vector en donde se escribirá.
 *  \param[in] n Es el número natural que se escribirá.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_natural(PdsBVector *BVector,PdsBaNatural n)
{
	PdsBaNatural i,d,t2;
	PdsBaByte t1;

	if(BVector==NULL)			return FALSE;

	d=256;
	t2=n;
	for(i=0;i<BVector->Nbytes;i++)
	{
		t1=(PdsBaByte)(t2%d);
		t2=t2/d;
		BVector->V[i]=t1;
	}

	t2=(BVector->Nbytes*8-BVector->Nel)%8;
	if(t2!=0)
	for(i=0;i<t2;i++)
	{
		BVector->V[BVector->Nbytes-1]=BVector->V[BVector->Nbytes-1]&(~(1<<(7-i)));
	}

	return TRUE;
}


/** \fn int pds_bvector_get_byte(const PdsBVector *BVector,PdsBaNatural n,PdsBaByte *m)
 *  \brief Obtiene el byte de la posición n del vector BVector y lo carga en m.
 *  \param[in] BVector Es el vector en donde se pedirá el byte.
 *  \param[in] n Es el número de byte a pedir.
 *  \param[out] m Es el byte pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_get_byte(const PdsBVector *BVector,PdsBaNatural n,PdsBaByte *m)
{
	if(BVector==NULL)	return FALSE;
	if(n >=BVector->Nbytes)	return FALSE;
	if(n < 0)		return FALSE;		

	*m=BVector->V[n];
	return TRUE;
}


/** \fn int pds_bvector_set_byte(PdsBVector *BVector,PdsBaNatural n,PdsBaByte m)
 *  \brief Escribe en el byte de la posición n del vector BVector, desde m.
 *  \param[in,out] BVector Es el vector en donde se escribirá el byte.
 *  \param[in] n Es el número de byte a escribir.
 *  \param[in] m Es el byte a escribir.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_byte(PdsBVector *BVector,PdsBaNatural n,PdsBaByte m)
{
	PdsBaNatural i,k;
	PdsBaByte B;

	if(BVector==NULL)	return FALSE;
	if(n >=BVector->Nbytes)	return FALSE;
	if(n < 0)		return FALSE;		

	k=(BVector->Nbytes*8-BVector->Nel)%8;
	B=0;
	for(i=0;i<k;i++)	B=B|(1<<(7-i));

	BVector->V[n]=m&(~B);
	return TRUE;
}


/** \fn int pds_bvector_get_bit(const PdsBVector *BVector,PdsBaNatural n,PdsBaBit *m)
 *  \brief Obtiene el bit de la posición n del vector BVector y lo carga en m.
 *  \param[in] BVector Es el vector en donde se pedirá el bit.
 *  \param[in] n Es el número de bit a pedir.
 *  \param[out] m Es el bit pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_get_bit(const PdsBVector *BVector,PdsBaNatural n,PdsBaBit *m)
{
	PdsBaNatural t2;
	PdsBaByte    t1;
	
	*m=0;
	if(BVector==NULL)	return FALSE;
	if(n >=BVector->Nel)	return FALSE;
	if(n < 0)		return FALSE;		

	t2=(n/8);
	t1=(PdsBaByte)(n%8);

	*m=(((BVector->V[t2])&(1<<t1))>>t1);
	return TRUE;
}


/** \fn int pds_bvector_set_bit(PdsBVector *BVector,PdsBaNatural n,PdsBaBit m)
 *  \brief Escribe el bit de la posición n del vector BVector y lo carga desde m.
 *  \param[in,out] BVector Es el vector en donde se escribirá el bit.
 *  \param[in] n Es el número de bit a escribir.
 *  \param[int] m Es el bit escrito.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_bit(PdsBVector *BVector,PdsBaNatural n,PdsBaBit m)
{
	PdsBaNatural t2;
	PdsBaByte    t1;
	
	if(BVector==NULL)	return FALSE;
	if(m > 1)		return FALSE;
	if(n >=BVector->Nel)	return FALSE;
	if(n < 0)		return FALSE;		

	t2=(n/8);
	t1=(PdsBaByte)(n%8);

	BVector->V[t2]=BVector->V[t2]&(~(1<<t1));
	BVector->V[t2]=BVector->V[t2]|(m<<t1);
	return TRUE;
}


/** \fn int pds_bvector_weight_bvector(const PdsBVector *BVector,PdsBaNatural *n)
 *  \brief Retorna el peso del vector binario BVector.
 *  \param[in] BVector Es un vector binario.
 *  \param[out] n El peso del vector binario BVector.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: El vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_weight_bvector(const PdsBVector *BVector,PdsBaNatural *n)
{
	PdsBaNatural i,j,k;
	PdsBaByte Byte;

	*n=0;

	if(BVector==NULL)			return FALSE;

	for(i=0,k=0;i<BVector->Nbytes;i++)
	{
		Byte=BVector->V[i];
		for(j=0;(j<8)&&(k<BVector->Nel);j++,k++)
		{
			*n=*n+((Byte&(1<<j))>>j);
		}
	}
	return TRUE;
}


/** \fn int pds_bvector_mean_bvector(const PdsBVector *BVector,PdsBaReal *m)
 *  \brief Retorna el valor medio del vector binario BVector.
 *  \param[in] BVector Es un vector binario.
 *  \param[out] m El valor medio del vector binario BVector.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: El vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_mean_bvector(const PdsBVector *BVector,PdsBaReal *m)
{
	PdsBaNatural i,j,k,n;
	PdsBaByte Byte;

	*m=0;
	if(BVector==NULL)			return FALSE;


	for(i=0,k=0,n=0;i<BVector->Nbytes;i++)
	{
		Byte=BVector->V[i];
		for(j=0;(j<8)&&(k<BVector->Nel);j++,k++)
		{
			n=n+((Byte&(1<<j))>>j);
		}
	}

	*m=n/(BVector->Nel*1.0);

	return TRUE;
}


/** \fn pds_bvector_xor_bvector(PdsBVector *BVector1,const PdsBVector *BVector2)
 *  \brief realiza a operación:  BVector1 = BVector1 XOR BVector2;
 *  \param[out] BVector1 Es un vector binario.
 *  \param[in] BVector2 Es un vector binario.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: la cantidad de
 *  elementos es distinta o algun vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_xor_bvector(PdsBVector *BVector1,const PdsBVector *BVector2)
{
	PdsBaNatural i;

	if(BVector1==NULL)			return FALSE;
	if(BVector2==NULL)			return FALSE;
	if(BVector1->Nel!=BVector2->Nel)	return FALSE;	
	if(BVector1->Nbytes!=BVector2->Nbytes)	return FALSE;	

	for(i=0;i<BVector1->Nbytes;i++)
	{
		BVector1->V[i]=BVector1->V[i]^BVector2->V[i];
	}
	return TRUE;
}


/** \fn int pds_bvector_xor_bvectors(PdsBVector *BVector1,const PdsBVector *BVector2,const PdsBVector *BVector3)
 *  \brief realiza a operación:  BVector1 = BVector2 XOR BVector3;
 *  \param[out] BVector1 Es un vector binario.
 *  \param[in] BVector2 Es un vector binario.
 *  \param[in] BVector3 Es un vector binario.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: la cantidad de
 *  elementos es distinta o algun vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_xor_bvectors(PdsBVector *BVector1,const PdsBVector *BVector2,const PdsBVector *BVector3)
{
	PdsBaNatural i;

	if(BVector1==NULL)			return FALSE;
	if(BVector2==NULL)			return FALSE;
	if(BVector3==NULL)			return FALSE;
	if(BVector1->Nel!=BVector2->Nel)	return FALSE;	
	if(BVector2->Nel!=BVector3->Nel)	return FALSE;	
	if(BVector1->Nbytes!=BVector2->Nbytes)	return FALSE;	
	if(BVector2->Nbytes!=BVector3->Nbytes)	return FALSE;	

	for(i=0;i<BVector1->Nbytes;i++)
	{
		BVector1->V[i]=(BVector2->V[i])^(BVector3->V[i]);
	}
	return TRUE;
}



/** \fn int pds_bvector_cor_bvector(const PdsBVector *VectorX,const PdsBVector *VectorY, PdsBaReal *c)
 *  \brief Devuelve el valor del coeficiente de correlación muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cor(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{\sqrt{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)^2}}\sqrt{\sum_{i=0}^{Nel-1} {(Y_i -{\eta}_y)^2}}} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la correlación de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_bvector_cor_bvector(const PdsBVector *VectorX,const PdsBVector *VectorY, PdsBaReal *c)
{
	PdsBaReal mx,my,txy,tx,ty;
	PdsBaBit a,b;
	PdsBaNatural i,id;

	*c=0;

	if(VectorX==NULL)		return FALSE;
	if(VectorY==NULL)		return FALSE;
	if(VectorX->Nel!=VectorY->Nel)	return FALSE;

	id=pds_bvector_mean_bvector(VectorX,&mx);
	if(id==FALSE)			return FALSE;
	id=pds_bvector_mean_bvector(VectorY,&my);
	if(id==FALSE)			return FALSE;

	for(txy=0,tx=0,ty=0,i=0;i<VectorX->Nel;i++)
	{
		pds_bvector_get_bit (VectorX, i, &a);
		pds_bvector_get_bit (VectorY, i, &b);

		txy=txy+(a-mx)*(b-my);
		tx =tx +(a-mx)*(a-mx);
		ty =ty +(b-my)*(b-my);
	}

	(*c)=txy/sqrt(tx*ty);
	return TRUE;
}


/** \fn int pds_bvector_printf(const PdsBVector *BVector)
 *  \brief Imprime en pantalla todos los elementos del vector;
 *  \param[in] BVector Es el vector a mostrar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_printf(const PdsBVector *BVector)
{
	PdsBaNatural i,j,k;
	PdsBaByte Bit;


	if(BVector==NULL)	return FALSE;


	for(i=0,k=0;i<BVector->Nbytes;i++)
	{
		for(j=0;(j<8)&&(k<BVector->Nel);j++,k++)
		{
			Bit=(BVector->V[i]&(1<<j))>>j;
			printf("%d",Bit);
		}
		if(i!=(BVector->Nbytes-1))	printf(" ");
	}
	printf("\n");
	

	return TRUE;
}


/** \fn void pds_bvector_free(PdsBVector *BVector)
 *  \brief Libera un vector de tipo puntero PdsBVector.
 *  \param[in,out] BVector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBVectorGroup
 */
void pds_bvector_free(PdsBVector *BVector)
{
	if(BVector!=NULL)
	{
		free(BVector->V);
		free(BVector);
	}
}


/** \fn void pds_bvector_destroy(PdsBVector **BVector)
 *  \brief Libera un vector de tipo puntero PdsBVector.
 *  \param[in,out] BVector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBVectorGroup
 */
void pds_bvector_destroy(PdsBVector **BVector)
{
	if((*BVector)!=NULL)
	{
		free((*BVector)->V);
		free(*BVector);
		(*BVector)=NULL;
	}
}

