/*
 * pdsbmatrix.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsbmatrix.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 *  \brief Funciones que trabajan con vectores binarios.
 *  
 *  Estas funciones trabajan con un vector binario de la forma:<br>
 *  \image html pdsbmatrix.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSBMATRIX_H__
#define __PDSBMATRIX_H__

#ifdef __cplusplus
extern "C" {
#endif 


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

#include <stdio.h>
#include <pds/pdsbaglobal.h>

/** \defgroup PdsBMatrixGroup Funciones del módulo PdsBMatrix.
 * @{
 */


/*! \struct PdsBMatrix
 *  \brief La estructura tipo  PdsBMatrix .
 *  Esta estructura genera una matriz binaria de Nel bits.
 *  Para usar incluir pds/pdsbmatrix.h.
 *  \image html pdsbmatrix.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 *  \ingroup PdsBMatrixGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	/*! Numero de Lineas. */
	PdsBaNatural Nlin;
	/*! Número de bloques de 8bits en cada linea..*/
	PdsBaNatural Nbytes;
	/*! Número de columnas. */
	PdsBaNatural Ncol;	
	/*! Matriz binaria. */
	PdsBaByte **M;
}PdsBMatrix;


/** \fn PdsBMatrix *pds_bmatrix_new(PdsBaNatural Nlin,PdsBaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsBMatrix.
 *  \param[in] Nlin Es el número de lineas de la matriz binaria.
 *  \param[in] Ncol Es el número de columnas de la matriz binaria.
 *  \return Un puntero a la matriz de tipo PdsBMatrix, o NULL si hubo un error.
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix *pds_bmatrix_new(PdsBaNatural Nlin,PdsBaNatural Ncol);


/** \fn PdsBMatrix *pds_bmatrix_new_bmatrix(const PdsBMatrix *M)
 *  \brief Crea una matriz de tipo PdsBMatrix que es una copia de outra matriz.
 *  \param[in] M Matriz de origen.
 *  \return Un puntero a una matriz de tipo PdsBMatrix, o NULL si hubo un error.
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix *pds_bmatrix_new_bmatrix(const PdsBMatrix *M);


/** \fn int pds_bmatrix_get_bit(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit *m)
 *  \brief Obtiene el bit de la posición n del vector BMatrix y lo carga en m.
 *  \param[in] BMatrix Es la matriz en donde se pedirá el bit.
 *  \param[in] lin Es la fila del bit a pedir.
 *  \param[in] col Es la columna del bit a pedir.
 *  \param[out] m Es el bit pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o lin/col fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_get_bit(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit *m);


/** \fn int pds_bmatrix_set_bit(PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit value)
 *  \brief Escribe el valor binario value en la posición (lin,col) de la matriz BMatrix.
 *  \param[in,out] BMatrix Es la matriz en donde se escribirá el bit.
 *  \param[in] lin Es el número de fila del bit a escribir.
 *  \param[in] col Es el número de columna del bit a escribir.
 *  \param[in] value Es el bit escrito.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o lin/col fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_set_bit(PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit value);


/** \fn int pds_bmatrix_xor_row(PdsBMatrix *BMatrix,PdsBaNatural row1,PdsBaNatural row2)
 *  \brief Ejejcuta la siguiente operacion BMatrix[row1]=BMatrix[row1] XOR BMatrix[row2].
 *  \param[in,out] BMatrix Es la matriz en operación.
 *  \param[in] row1 Primera linea.
 *  \param[in] row2 Segunda linea.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o row1/row2 fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_xor_row(PdsBMatrix *BMatrix,PdsBaNatural row1,PdsBaNatural row2);


/** \fn int pds_bmatrix_row_weight(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural *m)
 *  \brief Retorna la cantidad de unos en la linea lin de la matriz BMatrix.
 *  \param[in] BMatrix Es la matriz en anilisis.
 *  \param[in] lin Linea a contar.
 *  \param[out] m Es la cantidad de unos en la linea.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_row_weight(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural *m);


/** \fn int pds_bmatrix_find_first_one(PdsBMatrix *M,PdsBaNatural j)
 *  \brief Encuentra una linea que inicie en 1 en la columna "j" 
 *  a partir de la fila "j"(inclusive). La primera fila que encuentre 
 *  la coloca en la fila j. 
 *  \param[in,out] M Es la matriz en anilisis.
 *  \param[in] j Columna a buscar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: M==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_find_first_one(PdsBMatrix *M,PdsBaNatural j);


/** \fn int  pds_bmatrix_set_first_zero(PdsBMatrix *M,PdsBaNatural j)
 *  \brief Obliga con XOR de la linea j que todos los vectores abajo 
 *  y arriba de la fila j sean 0 en la columna j.
 *  \param[in,out] M Es la matriz en anilisis.
 *  \param[in] j Linea a ejecutar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: M==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int  pds_bmatrix_set_first_zero(PdsBMatrix *M,PdsBaNatural j);


/** \fn int pds_bmatrix_column_weight(const PdsBMatrix *BMatrix,PdsBaNatural col,PdsBaNatural *m)
 *  \brief Retorna la cantidad de unos en la columna col de la matriz BMatrix.
 *  \param[in] BMatrix Es la matriz en anilisis.
 *  \param[in] col Columna a contar.
 *  \param[out] m Es la cantidad de unos en la columna.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_column_weight(const PdsBMatrix *BMatrix,PdsBaNatural col,PdsBaNatural *m);


/** \fn PdsBMatrix * pds_bmatrix_systematize(const PdsBMatrix *S,PdsBaNatural *pos)
 *  \brief Retorna una matriz systemática o hasta donde se pueda sistematizar, carga 
 *  en pos la linea del ultimo bit de la matriz identidad. Para garantizar una matriz
 *  sistemática *pos tiene que ser igual a S->Nlin-1.
 *  \param[in] S Es la matriz en análisis.
 *  \param[out] pos La linea del ultimo bit de la matriz identidad.
 *  \return Retorna una matriz systemática. Retorna NULL si no fue posible crear la matriz
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix * pds_bmatrix_systematize(const PdsBMatrix *S,PdsBaNatural *pos);


/** \fn int pds_bmatrix_alist_save(const PdsBMatrix *m,const char *filename)
 *  \brief Escribe la matriz en un archivo con nombre filename en formato Alist.
 *  <a href="http://www.inference.phy.cam.ac.uk/mackay/codes/alist.html">Mackay</a>, 
 *  \param[in] m Es la matriz a gruadar.
 *  \param[in] filename Es el nombre del archivo donde será guardada la matriz.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_alist_save(const PdsBMatrix *m,const char *filename);


/** \fn int pds_bmatrix_is_zero(const PdsBMatrix *BMatrix, PdsBaNatural Lin)
 *  \brief Averigua si la matriz es nula desde la linea Lin (incluindo) hasta el final.
 *  \param[in] BMatrix La matriz a consultar.
 *  \param[in] Lin La linea inicial a consultar.
 *  \return TRUE si esta lleno de zeros desde la linea pos o FALSE sino. Si la
 *  matriz es nula también retorna FALSE.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_is_zero(const PdsBMatrix *BMatrix, PdsBaNatural Lin);


/** \fn int pds_bmatrix_printf(const PdsBMatrix *BMatrix)
 *  \brief Imprime en pantallala la matriz BMatrix.
 *  \param[in] BMatrix La matriz a imprimir.
 *  \return TRUE si todo fue bien o FALSE sino. Ejem: BMatrix==NULL
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_printf(const PdsBMatrix *BMatrix);


/** \fn void pds_bmatrix_free(PdsBMatrix *BMatrix)
 *  \brief Libera una matriz de tipo puntero PdsBMatrix.
 *  \param[in,out] BMatrix El puntero matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBMatrixGroup
 */
void pds_bmatrix_free(PdsBMatrix *BMatrix);


/** \fn void pds_bmatrix_destroy(PdsBMatrix **BMatrix)
 *  \brief Libera una matriz de tipo puntero PdsBMatrix y ademas carga NULL en el puntero.
 *  \param[in,out] BMatrix La direccion del puntero matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBMatrixGroup
 */
void pds_bmatrix_destroy(PdsBMatrix **BMatrix);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


