/*
 * pdscvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <pds/pdscvector.h>
#include <pds/pdscn.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsCVector                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsCVector *pds_cvector_new(PdsCaNatural Nel)
 *  \brief Crea un vector de tipo PdsCVector e inicia con cero todos los elementos.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsCVector.
 *  \ingroup PdsCVectorGroup
 */
PdsCVector *pds_cvector_new(PdsCaNatural Nel)
{
	PdsCVector *Vector=NULL;

	Vector=(PdsCVector *)calloc(1,sizeof(PdsCVector));
	if(Vector==NULL)	return NULL;

	Vector->Nel=Nel;
	Vector->V=(PdsComplex *)calloc(Vector->Nel,sizeof(PdsComplex));
	if(Vector->V==NULL) 
	{
		free(Vector);
		return NULL;
	}

	return Vector;
}


/** \fn PdsCVector *pds_cvector_new_cvector(const PdsCVector *VecSrc)
 *  \brief Crea un vector de tipo PdsCVector usando como fuente el vector VecSrc.
 *  \param[in] VecSrc Es el vector imagen que se usará como fuente.
 *  \return Un puntero al vector de tipo PdsCVector, retorna NULL en caso de error.
 *  \ingroup PdsCVectorGroup
 */
PdsCVector *pds_cvector_new_cvector(const PdsCVector *VecSrc)
{
	PdsCVector *Vector=NULL;
	PdsCaNatural i;

	if(VecSrc==NULL)	return NULL;

	Vector=(PdsCVector *)calloc(1,sizeof(PdsCVector));
	if(Vector==NULL)	return NULL;

	Vector->Nel=VecSrc->Nel;
	Vector->V=(PdsComplex *)calloc(Vector->Nel,sizeof(PdsComplex));
	if(Vector->V==NULL) 
	{
		free(Vector);
		return NULL;
	}
	else
	{
		for(i=0;i<Vector->Nel;i++)	Vector->V[i]=VecSrc->V[i];
	}

	return Vector;
}


/** \fn PdsCVector *pds_cvector_new_vector(const PdsVector *VecSrc)
 *  \brief Crea un vector de tipo PdsCVector usando como fuente el vector VecSrc
 *  y lo copia en la parte real del vector creado.
 *  \param[in] VecSrc Es el vector imagen que se usará como fuente.
 *  \return Un puntero al vector de tipo PdsCVector, retorna NULL en caso de error.
 *  \ingroup PdsCVectorGroup
 */
PdsCVector *pds_cvector_new_vector(const PdsVector *VecSrc)
{
	PdsCVector *Vector=NULL;
	PdsCaNatural i;

	if(VecSrc==NULL)	return NULL;

	Vector=(PdsCVector *)calloc(1,sizeof(PdsCVector));
	if(Vector==NULL)	return NULL;

	Vector->Nel=VecSrc->Nel;
	Vector->V=(PdsComplex *)calloc(Vector->Nel,sizeof(PdsComplex));
	if(Vector->V==NULL) 
	{
		free(Vector);
		return NULL;
	}
	else
	{
		for(i=0;i<Vector->Nel;i++)	Vector->V[i].Real=VecSrc->V[i];
	}

	return Vector;
}


/** \fn int pds_cvector_init_complex(PdsCVector *Vector,PdsComplex Val)
 *  \brief Inicia el vector de tipo puntero PdsCVector. <br><b>Vector=Val</b>.
 *  \param[in,out] Vector El vector a iniciar.
 *  \param[in] Val Es el valor complejo inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_complex(PdsCVector *Vector,PdsComplex Val)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=Val;
	}
	
	return TRUE;
}


/** \fn int pds_cvector_init_real(PdsCVector *Vector,PdsCaReal Val)
 *  \brief Inicia la parte real del vector de tipo puntero PdsCVector. <br><b>Vector.Real=Val</b>.
 *  \param[in,out] Vector El vector a iniciar.
 *  \param[in] Val Es el valor real inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_real(PdsCVector *Vector,PdsCaReal Val)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i].Real=Val;
	}
	
	return TRUE;
}


/** \fn int pds_cvector_init_imag(PdsCVector *Vector,PdsCaReal Val)
 *  \brief Inicia la parte imaginaria del vector de tipo puntero PdsCVector. <br><b>Vector.Imag=Val</b>.
 *  \param[in,out] Vector El vector a iniciar.
 *  \param[in] Val Es el valor imaginario inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_imag(PdsCVector *Vector,PdsCaReal Val)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i].Imag=Val;
	}
	
	return TRUE;
}


/** \fn int pds_cvector_init_cvector(PdsCVector *Vector,const PdsCVector *VecSrc)
 *  \brief Inicia el vector de tipo puntero PdsCVector con otro vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=VecSrc</b>.
 *  \param[in,out] Vector Es el vector a iniciar.
 *  \param[in] VecSrc Es el vector fuente de la copia.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_cvector(PdsCVector *Vector,const PdsCVector *VecSrc)
{
	PdsCaNatural i;
	PdsCaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_cvector_init_rvector(PdsCVector *Vector,const PdsVector *VecSrc)
 *  \brief Inicia la parte real del vector de tipo puntero PdsCVector con un 
 *  vector de tipo PdsVector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector.Real=VecSrc</b>.
 *  \param[in,out] Vector Es el vector a iniciar la parte real.
 *  \param[in] VecSrc Es el vector fuente de la copia.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_rvector(PdsCVector *Vector,const PdsVector *VecSrc)
{
	PdsCaNatural i;
	PdsCaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i].Real=VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_cvector_init_ivector(PdsCVector *Vector,const PdsVector *VecSrc)
 *  \brief Inicia la parte imaginaria del vector de tipo puntero PdsCVector con un 
 *  vector de tipo PdsVector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector.Imag=VecSrc</b>.
 *  \param[in,out] Vector Es el vector a iniciar la parte imaginaria.
 *  \param[in] VecSrc Es el vector fuente de la copia.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_init_ivector(PdsCVector *Vector,const PdsVector *VecSrc)
{
	PdsCaNatural i;
	PdsCaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i].Imag=VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_cvector_printf(const PdsCVector *Vector)
 *  \brief Imprime en pantalla un vector de tipo puntero PdsCVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_printf(const PdsCVector *Vector)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	//printf("\n");
	for(i=0;i<Vector->Nel;i++)
	{
		printf("%e",Vector->V[i].Real);
		
		if(i==(Vector->Nel-1))	printf("\n");
		else			printf("\t");
	}
	for(i=0;i<Vector->Nel;i++)
	{
		printf("%e",Vector->V[i].Imag);
		
		if(i==(Vector->Nel-1))	printf("\n");
		else			printf("\t");
	}
	return TRUE;
}


/** \fn int pds_cvector_fprintf(const PdsCVector *Vector,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd un vector de tipo puntero PdsCVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \param[in,out] fd El puntero al archivo.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_fprintf(const PdsCVector *Vector,FILE *fd)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	//fprintf(fd,"\n");
	for(i=0;i<Vector->Nel;i++)
	{
		fprintf(fd,"%e",Vector->V[i].Real);
		
		if(i==(Vector->Nel-1))	fprintf(fd,"\n");
		else			fprintf(fd,"\t");
	}
	for(i=0;i<Vector->Nel;i++)
	{
		fprintf(fd,"%e",Vector->V[i].Imag);
		
		if(i==(Vector->Nel-1))	fprintf(fd,"\n");
		else			fprintf(fd,"\t");
	}
	return TRUE;
}


/** \fn int pds_cvector_get_complex(const PdsCVector *Vector, PdsCaNatural x, PdsComplex *m)
 *  \brief Devuelve el valor complejo en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in] Vector El vector en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor complejo en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0+0i en m, en caso de error de vector nulo carga 0+0i en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_get_complex(const PdsCVector *Vector, PdsCaNatural x, PdsComplex *m)
{
	(*m).Real=0;
	(*m).Imag=0;

	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	
	{
		(*m).Real=Vector->V[x].Real;
		(*m).Imag=Vector->V[x].Imag;
	}
	else				
	{
		(*m).Real=0;
		(*m).Imag=0;
	}

	return TRUE;
}


/** \fn int pds_cvector_set_complex(PdsCVector *Vector, PdsCaNatural x, PdsComplex m)
 *  \brief Escribe el valor m en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor complejo en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_set_complex(PdsCVector *Vector, PdsCaNatural x, PdsComplex m)
{
	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	Vector->V[x]=m;

	return TRUE;
}


/** \fn int pds_cvector_set_real(PdsCVector *Vector, PdsCaNatural x, PdsCaReal m)
 *  \brief Escribe el valor m solo en la parte real de la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m La parte real a escribir en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_set_real(PdsCVector *Vector, PdsCaNatural x, PdsCaReal m)
{
	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	Vector->V[x].Real=m;

	return TRUE;
}


/** \fn int pds_cvector_set_imag(PdsCVector *Vector, PdsCaNatural x, PdsCaReal m)
 *  \brief Escribe el valor m solo en la parte imaginaria de la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m La parte imaginaria a escribir en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_set_imag(PdsCVector *Vector, PdsCaNatural x, PdsCaReal m)
{
	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	Vector->V[x].Imag=m;

	return TRUE;
}


/** \fn int pds_cvector_conjugate(PdsCVector *Vector)
 *  \brief Encuentra el conjugado del vector Vector. <br><b>Vector=CONJ(Vector)</b>.
 *  \param[in,out] Vector Es el vector a obtener el conjugado.
 *  \return TRUE si todo fue bien o FALSE si no, por ejemplo vector NULL o
 *  tamanhos distintos.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_conjugate(PdsCVector *Vector)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i].Imag=-Vector->V[i].Imag;
	}
	
	return TRUE;
}


/** \fn int pds_cvector_conjugate_cvector(const PdsCVector *Vector,PdsCVector *VecDest)
 *  \brief Encuentra el conjugado del vector Vector, y el resultado es copiado
 *  en el vector VecDest. <br><b>VecDest=CONJ(Vector)</b>.
 *  \param[in] Vector Es el vector a obtener el conjugado.
 *  \param[in,out] VecDest Es el conjugado del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no, por ejemplo vector NULL o
 *  tamanhos distintos.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_conjugate_cvector(const PdsCVector *Vector,PdsCVector *VecDest)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	if(VecDest==NULL)	return FALSE;

	if(Vector->Nel != VecDest->Nel)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		VecDest->V[i].Real= Vector->V[i].Real;
		VecDest->V[i].Imag=-Vector->V[i].Imag;
	}
	
	return TRUE;
}


/** \fn int pds_cvector_modulus(const PdsCVector *Vector,PdsVector *VecDest)
 *  \brief Encuentra el Modulo del vector Vector, y el resultado es copiado
 *  en el vector VecDest. <br><b>VecDest=|Vector|</b>.
 *  \param[in] Vector Es el vector a obtener el modulo.
 *  \param[in,out] VecDest Es el módulo del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no, por ejemplo vector NULL o
 *  tamanhos distintos.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_modulus(const PdsCVector *Vector,PdsVector *VecDest)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	if(VecDest==NULL)	return FALSE;

	if(Vector->Nel != VecDest->Nel)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		VecDest->V[i]=sqrt(Vector->V[i].Real*Vector->V[i].Real+Vector->V[i].Imag*Vector->V[i].Imag);
	}
	
	return TRUE;
}


/** \fn int pds_cvector_modulus2(const PdsCVector *Vector,PdsVector *VecDest)
 *  \brief Encuentra el Modulo^2 del vector Vector, y el resultado es copiado
 *  en el vector VecDest. <br><b>VecDest=|Vector|^2</b>.
 *  \param[in] Vector Es el vector a obtener el modulo^2.
 *  \param[in,out] VecDest Es el módulo^2 del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no, por ejemplo vector NULL o
 *  tamanhos distintos.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_modulus2(const PdsCVector *Vector,PdsVector *VecDest)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	if(VecDest==NULL)	return FALSE;

	if(Vector->Nel != VecDest->Nel)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		VecDest->V[i]=(Vector->V[i].Real*Vector->V[i].Real+Vector->V[i].Imag*Vector->V[i].Imag);
	}
	
	return TRUE;
}


/** \fn int pds_cvector_phase(const PdsCVector *Vector,PdsVector *VecDest)
 *  \brief Encuentra la fase de Z=X+Yi phase=atan2(Y,X) del vector Vector, y el 
 *  resultado es copiado en el vector VecDest. <br><b>VecDest=Phase{Vector}</b>.
 *  \param[in,out] Vector Es el vector a obtener la fase.
 *  \param[in] VecDest Es la fase del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no, por ejemplo vector NULL o
 *  tamanhos distintos.
 *  \ingroup PdsCVectorGroup
 */
int pds_cvector_phase(const PdsCVector *Vector,PdsVector *VecDest)
{
	PdsCaNatural i;

	if(Vector==NULL)	return FALSE;
	if(VecDest==NULL)	return FALSE;

	if(Vector->Nel != VecDest->Nel)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		VecDest->V[i]=atan2(Vector->V[i].Imag,Vector->V[i].Real);
	}
	
	return TRUE;
}


/** \fn void pds_cvector_free(PdsCVector *Vector)
 *  \brief Libera un vector de tipo puntero PdsCVector.
 *  \param[in,out] Vector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsCVectorGroup
 */
void pds_cvector_free(PdsCVector *Vector)
{

	free(Vector->V);

	free(Vector);
}


/** \fn void pds_cvector_destroy(PdsCVector **Vector)
 *  \brief Libera un vector de tipo puntero PdsCVector, y limpia el puntero con NULL.
 *  \param[in,out] Vector El vector a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsCVectorGroup
 */
void pds_cvector_destroy(PdsCVector **Vector)
{

	free((*Vector)->V);

	free(*Vector);
	(*Vector)=NULL;
}

