/*
 * pdsbsc.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsbsc.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 *  \brief Funciones que trabajan con un modelos de canal BSC.
 *  
 *  Estas funciones trabajan con un canal BSC así:.<br>
 *  \image html image.png "Canal BSC"
 *  \b X es la entrada real del canal, \b Z es una variable aleatoria Gaussiana.
 *  de media cero y varianza Sigma^2 e \b Y es la salida real del canal.
 */

#ifndef __PDSBSC_H__
#define __PDSBSC_H__

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdscmglobal.h>
#include <pds/pdsuniform.h>
#include <pds/pdsbvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsBscGroup Funciones del módulo PdsBsc.
 * @{
 */


/*! \struct PdsBsc
 *  \brief La estructura tipo  PdsBsc .
 *  Esta estructura genera una canal BSC.
 *  Para usar incluir pds/pdsbsc.h.
 *  \ingroup PdsBscGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Una variable aleatoria Gaussiana. */
	PdsUniform *Z;

	/*! Probabilida de error del canal.*/
	PdsCmReal p;
}PdsBsc;


/** \fn PdsBsc *pds_bsc_new(PdsCmReal p)
 *  \brief Crea un canal de tipo PdsBsc.
 *  \param[in] p La probabilidad de error del canal.
 *  \return Un puntero al canal de tipo PdsBsc, o NULL si hubo un error.
 *  \ingroup PdsBscGroup
 */
PdsBsc *pds_bsc_new(PdsCmReal p);


/** \fn int pds_bsc_evaluate(PdsBsc *Channel,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput)
 *  \brief Evalua un vector binario en un canal BSC.
 *  \param[in,out] Channel Canal BSC.
 *  \param[in] BVectorInput Es el vector binario a la entrada del canal BSC.
 *  \param[out] BVectorOutput Es el vector binario a la salida del canal BSC.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBscGroup
 */
int pds_bsc_evaluate(PdsBsc *Channel,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput);


/** \fn void pds_bsc_free(PdsBsc *Channel)
 *  \brief Libera un canal de tipo puntero PdsBsc.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscGroup
 */
void pds_bsc_free(PdsBsc *Channel);


/** \fn void pds_bsc_destroy(PdsBsc **Channel)
 *  \brief Libera un canal de tipo puntero PdsBsc y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscGroup
 */
void pds_bsc_destroy(PdsBsc **Channel);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif
/* FPRNEURON_H */ 

